import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';
import channelDirectoryScenario, { channelNames } from 'web-client/mirage/scenarios/channel-directory';
import ChannelDirectoryPage from 'web-client/tests/pages/channel-directory';
import run from 'ember-runloop';
import { stubLogin } from 'web-client/tests/helpers/stub-login';

let page;

moduleForAcceptance('Acceptance-Channel Directory', {
  beforeEach() {
    page = ChannelDirectoryPage.create();
    channelDirectoryScenario(this.server);

    this.experiments.use({
      RECOMMENDED_CHANNELS: 'yes',
      WATCH_NOW_RIGHTBAR: 'control'
    });
    const watchedChannelsService = this.owner.lookup('service:watched-channels');
    const watchedStream = this.server.schema.streams.where({ name: channelNames[0] }).models[0];
    watchedChannelsService.addChannel(watchedStream.channel.name);
  },

  afterEach() {
    // addChannel creates records in the store, above, which leaks into other
    // tests unless we clean it up :|
    let store = this.owner.lookup('service:store');
    store.peekAll('watched-channel').forEach(channel => {
      run(channel, 'destroy');
    });
  }
});

test('language directory filter appears when in control group', function(assert) {
  this.experiments.use({'LANGUAGE_DIRECTORY_FILTER': 'control'});

  assert.expect(1);

  visit(page.url());

  andThen(function () {
    assert.equal(find(page.languageFilter()).length, 1, 'language filter is not visible');
  });
});

test('language directory filter appears when in experiment group', function(assert) {
  assert.expect(2);

  this.experiments.use({'LANGUAGE_DIRECTORY_FILTER': 'full'});

  visit(page.url());

  andThen(function () {
    assert.equal(find(page.languageFilter()).length, 1, 'language filter is visible for full');
  });

  this.experiments.use({'LANGUAGE_DIRECTORY_FILTER': 'scroll'});

  visit(page.url());

  andThen(function () {
    assert.equal(find(page.languageFilter()).length, 1, 'language filter is visible for scroll');
  });
});

test('visiting channel directory page', function (assert) {
  assert.expect(6);

  visit(page.url());

  andThen(function () {
    assert.equal(currentURL(), '/directory/all');
    assert.equal(currentRouteName(), 'directory.channels.all');
    assert.equal(find(page.videoThumbs()).length, 6, 'has 6 video thumbs');
    assert.equal(page.canonicalUrl(), 'https://www.twitch.tv/directory/all', 'Canonical URL present');
    assert.equal(page.headTitle(), 'Top Channels - Twitch', 'has title tag in head');
    assert.equal(find(page.navigationTabs()).length, 5, 'has 5 navigation tabs');
  });
});

test('clicking channel thumbnail takes you to channel page', function (assert) {
  assert.expect(2);

  visit(page.url());

  click(page.thumbLink(channelNames[0]));

  andThen(() => {
    assert.equal(currentURL(), `/${channelNames[0]}`, 'transitioned');
    assert.equal(currentRouteName(), 'channel.index.index');
  });
});

test('recommended channels has correct HREF', function (assert) {
  assert.expect(1);

  visit(page.url());

  andThen(function () {
    assert.equal(page.recommendedChannelHref(), `/${channelNames[1]}`, 'href set for watched channel');
  });
});

test('clicking recommended channels takes you to channel page', function (assert) {
  assert.expect(2);

  visit(page.url());

  click(page.recommendedChannelLink());

  andThen(() => {
    assert.equal(currentURL(), `/${channelNames[1]}`, 'transitioned');
    assert.equal(currentRouteName(), 'channel.index.index');
  });
});

test('clicking platforms tabs has expected elements', function (assert) {
  assert.expect(6);

  visit(page.url());

  // xbox one
  click(page.xboxOneTab());

  andThen(() => {
    assert.equal(currentURL(), '/directory/all/xb1', 'has correct route for xbox one');
    assert.equal(find(page.videoThumbs()).length, 1, 'has 1 video thumbs');
    assert.ok(page.hasHeaderText('Xbox One'));
  });

  // playstation 4
  visit(page.url());
  click(page.psFourTab());

  andThen(() => {
    assert.equal(currentURL(), '/directory/all/ps4', 'has correct route for ps4');
    assert.equal(find(page.videoThumbs()).length, 3, 'has 3 video thumbs');
    assert.ok(page.hasHeaderText('PlayStation 4'));
  });
});

moduleForAcceptance('Acceptance-Channel Directory w/ new user', {
  beforeEach() {
    page = ChannelDirectoryPage.create();
    channelDirectoryScenario(this.server);
    stubLogin();
  }
});

test('visiting channel directory page shows following FTUX in sidebar', function (assert) {
  assert.expect(4);

  visit(page.url());

  andThen(function () {
    assert.equal(currentURL(), '/directory/all');
    assert.equal(currentRouteName(), 'directory.channels.all');

    // Following FTUX
    assert.ok(
      page.hasRightSidebarText(`Nobody! You're following nobody!`),
      'prompts you are following nobody'
    );
    assert.ok(
      page.hasRightSidebarText(`When you follow a channel, they'll show up right here for easy access!`),
      'prompt to follow a channel'
    );
  });
});

test('visiting channel directory page shows whispers FTUX in sidebar', function (assert) {
  assert.expect(8);

  visit(page.url());

  andThen(function () {
    assert.equal(currentURL(), '/directory/all');
    assert.equal(currentRouteName(), 'directory.channels.all');

    // Whispers FTUX
    assert.ok(page.hasLeftSidebarText(`Add friends to see their online status and start whispering!`));
    assert.ok(page.hasLeftSidebarText(`Got it`));
  });

  click('a:contains(Got it)');

  andThen(function () {
    // Have not changed the page
    assert.equal(currentURL(), '/directory/all');
    assert.equal(currentRouteName(), 'directory.channels.all');

    // Whispers FTUX is removed
    assert.notOk(page.hasLeftSidebarText(`Add friends to see their online status and start whispering!`));
    assert.notOk(page.hasLeftSidebarText(`Got it`));
  });

});
