/* globals Twitch */
import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';
import channelScenario from 'web-client/mirage/scenarios/channel';
import ClipsPage from 'web-client/tests/pages/channel/clips';
import sinon from 'sinon';
import { CLIPS_TRACKING_LOCATION } from 'web-client/routes/channel/clips';
import { TEST_SELECTOR as clipsSortSelector, DROPDOWN_TEST_SELECTOR } from 'web-client/components/filter-bar/filters/clips-sort/component';
import { TEST_SELECTOR as sortOptionSelector } from 'web-client/components/clips/clips-sort-option/component';

const CHANNEL_NAME = 'twitch';

moduleForAcceptance('Acceptance | Channel | Clips', {
  beforeEach() {
    channelScenario(this.server, CHANNEL_NAME);
    let channel = this.server.schema.findBy('user', { name: CHANNEL_NAME });

    this.broadcaster = {
      id: channel._id,
      login: channel.login,
      display_name: channel.login,
      channel_url: `https://www.twitch.tv/${CHANNEL_NAME}`
    };
    this.page = ClipsPage.create({ name: CHANNEL_NAME });
    this.clipsSortSelector = `[data-test-selector="${clipsSortSelector}"]`;
    this.dropdownTestSelector = `[data-test-selector="${DROPDOWN_TEST_SELECTOR}"]`;
    this.sortOptionSelector = `[data-test-selector="${sortOptionSelector}"]`;
    this.emptyContainer = `[data-test-selector="clips-empty-container"]`;
  }
});

test('visiting the Clips page when the experiment is disabled', function(assert) {
  assert.expect(1);

  this.experiments.use({ CLIPS_FEEL_IT_COMING: 'control' });

  visit(this.page.url());

  andThen(() => {
    assert.equal(currentURL(), `/${CHANNEL_NAME}`, 'redirects to the channel page');
  });
});

test('visiting the Clips page when the experiment is enabled', function(assert) {
  assert.expect(4);
  sinon.spy(Twitch.tracking.funnel, 'trackPageView');

  this.server.createList('clip', 50, { broadcaster: this.broadcaster });
  this.experiments.use({ CLIPS_FEEL_IT_COMING: 'variant1' });

  visit(this.page.url());

  andThen(() => {
    assert.equal(currentURL(), `/${CHANNEL_NAME}/clips`, 'successfully land on the Clips page');
    assert.equal(Twitch.tracking.funnel.trackPageView.firstCall.args[0].location, CLIPS_TRACKING_LOCATION, 'tracks pageview location');
    assert.elementCount(this.page.clips(), 20, 'renders clip rows from fixture data');
    this.page.scrollDown();
  });

  andThen(() => {
    assert.elementCount(this.page.clips(), 40, 'renders more clips after scroll');
    Twitch.tracking.funnel.trackPageView.restore();
  });
});

test('when no Clips are available and the active sort is all-time popular', function(assert) {
  assert.expect(4);
  this.experiments.use({ CLIPS_FEEL_IT_COMING: 'variant1' });

  let trendingSortOptionSelector = `${this.dropdownTestSelector} ${this.sortOptionSelector}:eq(4)`;

  visit(this.page.url());
  click(this.clipsSortSelector);

  andThen(() => {
    click(trendingSortOptionSelector);
  });

  andThen(() => {
    assert.elementCount(this.page.clips(), 0, 'does not render any clips');

    assert.elementText('p[data-test-selector="clips-empty-title"]', 'No Clips Found', 'renders no clips found message');
    assert.elementText(
      '[data-test-selector="clips-help"]',
      `Learn how to make and share clips of ${CHANNEL_NAME}`,
      'shows help message'
    );
    assert.elementCount(`a[href="https://help.twitch.tv/customer/portal/articles/2442508-how-to-use-clips"]`, 1, 'links to help article');
  });
});

test('when no Clips are available and the active sort is not all-time popular', function(assert) {
  assert.expect(3);

  let activeSortOptionSelector = `${this.clipsSortSelector} ${this.sortOptionSelector}:first`;

  this.experiments.use({ CLIPS_FEEL_IT_COMING: 'variant1' });

  visit(this.page.url());

  andThen(() => {
    assert.elementText(
      `${this.emptyContainer} h5`,
      'No clips were created in this time period. Check out all popular clips from this channel instead.',
      'renders popular clips upsell text'
    );

    let popularClipButtonSelector = `${this.emptyContainer} button`;
    assert.elementText(popularClipButtonSelector, 'See all popular clips', 'renders button for popular clips');

    click(popularClipButtonSelector);
  });

  andThen(() => {
    assert.elementText(activeSortOptionSelector, 'Top all', 'active sort option is Popular (all-time)');
  });
});
