/* globals sendIRCMessage */
import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';
import channelScenario from 'web-client/mirage/scenarios/channel';
import followerScenario from 'web-client/mirage/scenarios/follower';
import followScenario from 'web-client/mirage/scenarios/follow';
import ChannelPage from 'web-client/tests/pages/channel/redesign';
import { lastPlayerInstance } from 'web-client/tests/helpers/fake-player';
import { initializeChat } from 'web-client/tests/helpers/chat';

let channelName = 'twitch';

moduleForAcceptance('Acceptance | channel-redesign/index', {
  beforeEach() {
    channelScenario(this.server, channelName);
    this.page = ChannelPage.create({ name: channelName });
    this.experiments.use({
      DETERMINATION: 'yes',
      CHANNEL_PAGE_REDESIGN: 'new',
      CHANNEL_LOGIN_CTA: 'yes'
    });
  },
  afterEach() {
    this.experiments.use({ CHANNEL_PAGE_REDESIGN: 'false' });
  }
});

test('visiting channel page and clicking a game name which would need encoding', function (assert) {
  assert.expect(5);

  let gameName = 'Pokémon Sun/Moon';

  this.server.create('game', { name: gameName });

  let channel = this.server.schema.db.channelEmberData.where({name: channelName})[0];

  this.server.schema.db.channelEmberData.update(channel.id, {game: gameName});

  visit(this.page.url());

  andThen(() => {
    assert.equal(currentURL(), `/${channelName}`);
    assert.equal(currentRouteName(), 'channel.index.index');

    assert.ok(this.page.metadataBox(), 'Metadata box rendered');
  });

  click('[data-test-selector=game-icon]');

  andThen(() => {
    assert.equal(currentURL(), `/directory/game/Pok%C3%A9mon%20Sun%2FMoon`, 'url is properly encoded');
    assert.equal(currentRouteName(), 'directory.game.index');
  });
});

test('visiting channel page', function (assert) {
  assert.expect(6);

  visit(this.page.url());

  andThen(() => {
    assert.equal(currentURL(), `/${channelName}`);
    assert.equal(currentRouteName(), 'channel.index.index');
    assert.equal(this.page.canonicalUrl(), `https://www.twitch.tv/${channelName}`, 'Canonical URL present');
    assert.equal(this.page.headTitle(), `${channelName} - Twitch`, 'has title tag in head');

    let $chat = find(this.page.chatContainer);
    assert.equal($chat.length, 1, '.js-rightcol-content .js-chat-container renders');
    assert.ok(this.page.metadataBox(), 'Metadata box rendered');
  });
});

test('interactivity tracking', function (assert) {
  assert.expect(2);

  visit(this.page.url());

  andThen(() => {
    initializeChat();

    // Invoke the player's "playing" event to trigger its "reportInteractive" method
    lastPlayerInstance.listeners['contentShowing'].forEach(function (listener) {
      listener();
    });
  });

  andThen(() => {
    assert.trackInteractivity('channel.index.index');
    assert.trackBenchmarkEvent('complete_transition');
  });
});

test('top bar; clicking followers takes you to the followers page', function (assert) {
  assert.expect(3);

  followerScenario(this.server);

  visit(this.page.url());

  andThen(() => {
    let link = find(this.page.followersButton);
    assert.ok(link.text().trim().match(/Followers\s+1/), 'shows followers link with count');
    click(link);
  });

  andThen(() => {
    assert.equal(currentURL(), '/twitch/followers');
    assert.equal(currentRouteName(), 'channel.followers');

    // TODO: These following lines are related to scrolling problems (with {{infinite-scroll}})
    // that we know about, but these break intermittently depending on user's screen size.
    // Commenting this check out for now while we investigate scrolling issues.

    // assert.ok(this.page.profileCardLink('follower'), 'follower rendered in profile card');
    // click(this.page.profileCardLink('follower'));
  });

  // andThen(() => {
  //   assert.equal(currentURL(), '/follower', 'redirected to follower channel url');
  //   assert.equal(currentRouteName(), 'channel.index.index', 'redirect to follower channel route');
  // });
});

test('top bar; clicking following takes you to the following page', function (assert) {
  assert.expect(3);

  followScenario(this.server, { name: 'CounterPit' });

  visit(this.page.url());

  andThen(() => {
    let link = find(this.page.followingButton);
    assert.ok(link.text().trim().match(/Following\s+1/), 'shows following link with count');
    click(link);
  });

  andThen(() => {
    assert.equal(currentURL(), '/twitch/following');
    assert.equal(currentRouteName(), 'channel.following');

    // TODO: These following lines are related to scrolling problems (with {{infinite-scroll}})
    // that we know about, but these break intermittently depending on user's screen size.
    // Commenting this check out for now while we investigate scrolling issues.

    // assert.ok(this.page.profileCardLink('CounterPit'), 'followed rendered in profile card');
    // click(this.page.profileCardLink('CounterPit'));
  });

  // andThen(() => {
    // assert.equal(currentURL(), '/CounterPit', 'redirected to followed channel url');
    // assert.equal(currentRouteName(), 'channel.index.index', 'redirect to followed channel route');
  // });
});

test('last broadcast notification is on the page', function(assert) {
  assert.expect(1);
  visit(this.page.url());
  andThen(() => {
    assert.equal(find('.js-channel-upsell').length, 1);
  });
});

test('last broadcast notification is not on the page in maintenance mode', function(assert) {
  this.owner.lookup('service:globals').set('isInMaintenanceMode', true);
  assert.expect(1);
  visit(this.page.url());
  andThen(() => {
    assert.equal(find('.js-channel-upsell').length, 0);
  });
});

test('last broadcast notification is not on the page when channel is hosting', function(assert) {
  assert.expect(1);
  channelScenario(this.server, 'extalix');
  visit(this.page.url());

  sendIRCMessage({
    command: 'HOSTTARGET',
    hostTarget: channelName,
    target: 'extalix'
  });

  andThen(() => {
    assert.equal(find('.js-channel-upsell').length, 0);
  });
});
