import { test } from 'qunit';
import moduleForAcceptance from '../helpers/module-for-acceptance';
import channelScenario from 'web-client/mirage/scenarios/channel';
import followerScenario from 'web-client/mirage/scenarios/follower';
import ChannelPage from '../pages/channel';
import ChannelRedesignPage from 'web-client/tests/pages/channel/redesign';
import { stubLogin } from '../helpers/stub-login';
import Mousetrap from 'mousetrap';

let channelName = 'twitch';
let page;

moduleForAcceptance('Acceptance | Channel', {
  beforeEach() {
    channelScenario(this.server, channelName);
    page = ChannelPage.create({ name: channelName });
  }
});

test('visiting channel page', function (assert) {
  assert.expect(5);

  followerScenario(this.server);
  visit(page.url());

  andThen(function () {
    assert.equal(currentURL(), `/${channelName}`);
    assert.equal(currentRouteName(), 'channel.index.index');
    assert.equal(page.canonicalUrl(), `https://www.twitch.tv/${channelName}`, 'Canonical URL present');
    assert.equal(page.headTitle(), `${channelName} - Twitch`, 'has title tag in head');
    assert.ok(page.showsStat('js-cn-tab-followers', 1), 'shows count of 1 follower');
  });
});

test('unable to see Clips tab when experiment is disabled', function (assert) {
  assert.expect(1);

  let redesignPage = ChannelRedesignPage.create({ name: channelName });
  this.experiments.use({ CLIPS_FEEL_IT_COMING: 'control' });

  visit(page.url());

  andThen(function () {
    assert.elementCount(redesignPage.clipsTabButton, 0);
  });
});

test('able to see Clips tab when experiment is enabled', function (assert) {
  assert.expect(1);

  let redesignPage = ChannelRedesignPage.create({ name: channelName });
  this.experiments.use({ CLIPS_FEEL_IT_COMING: 'variant1' });

  visit(page.url());

  andThen(function () {
    assert.elementCount(redesignPage.clipsTabButton, 1);
  });
});

test('logging in as a user shows the edit panel button in the channel when the panels are finished loading', function (assert) {
  assert.expect(1);
  let done = assert.async();

  stubLogin();
  visit(page.url());

  andThen(() => {
    if (find('.tw-tabs')) {
      click('.tw-tabs button:contains("Info")');
    }
  });

  andThen(() => {
    assert.equal(find(page.visibleEditPanel()).length, 1);
    done();
  });
});

test('chat sidebar renders', function(assert) {
  assert.expect(1);
  visit(page.url());

  andThen(() => {
    let $chat = find('.js-rightcol-content .js-chat-container');
    assert.equal($chat.length, 1, '.js-rightcol-content .js-chat-container renders');
  });
});

test('turning on theater mode hides wormhole-overlays', function(assert) {
  assert.expect(2);
  visit(page.url());

  andThen(() => {
    assert.equal($('#wormhole-overlays').is(':hidden'), false, 'user can see overlays');

    // enable theatre mode
    Mousetrap.trigger('alt+t');
  });

  andThen(() => {
    assert.equal($('#wormhole-overlays').is(':hidden'), true, 'overlays are hidden');
  });
});
