import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';
import channelScenario from 'web-client/mirage/scenarios/channel';
import ChannelPage from 'web-client/tests/pages/channel-feed';
import { stubLogin } from 'web-client/tests/helpers/stub-login';

let channelName = 'twitch';
let page;

moduleForAcceptance('Acceptance | Channel Feed', {
  beforeEach() {
    channelScenario(this.server, channelName);
    page = ChannelPage.create({ name: channelName });
    this.notify = this.owner.lookup('service:notify');
  }
});

test('visiting channel page should display feed', function (assert) {
  assert.expect(1);
  stubLogin({ login: channelName });

  let post = this.server.create('post');
  visit(page.url());

  andThen(() => {
    assert.ok(page.hasFeed(post.body));
  });
});

test('cannot add a blank post to a channel feed', function (assert) {
  assert.expect(3);
  stubLogin({ username: channelName });

  let postTitle = ' ';
  visit(page.url());

  assert.equal(this.notify._invocations.length, 0, 'there are no errors');

  fillIn(page.feedTextarea, postTitle);
  click(page.postFeedButton);

  andThen(() => {
    assert.equal(this.notify._invocations.length, 0, 'there are still no errors');
    let post = this.server.db.posts[0];
    assert.equal(post, undefined, 'Post not saved to DB');
  });
});

test('can add a post to a channel feed', function (assert) {
  assert.expect(2);
  stubLogin({ username: channelName });

  let postTitle = 'Hello, World!';
  visit(page.url());
  fillIn(page.feedTextarea, postTitle);
  click(page.postFeedButton);

  andThen(() => {
    let post = this.server.db.posts[0];
    assert.ok(page.hasFeed(postTitle));
    assert.ok(post, 'Post saved to DB');
  });
});

test('can remove a post from a channel feed', function (assert) {
  assert.expect(3);
  stubLogin({ username: channelName });

  let post = this.server.create('post');
  visit(page.url());

  andThen(() => {
    assert.ok(page.hasFeed(post.body, 'Post is on page'));
  });

  click(page.postModControls);
  click(page.deletePostMenu);
  click(page.confirmDeletePost);

  andThen(() => {
    let posts = this.server.db.posts;
    assert.equal(posts.length, 0, 'The post has been deleted from the DB.');
    assert.notOk(page.hasFeed(post.body));
  });
});

test('can add a comment', function (assert) {
  assert.expect(2);
  stubLogin({ username: channelName });
  this.server.create('post');

  let commentTitle = 'Best comment ever';

  visit(page.url());
  fillIn(page.commentTextarea, commentTitle);
  keyEvent(page.commentTextarea, 'keydown', 13);

  andThen(() => {
    let comment = this.server.db.comments[0];
    assert.ok(page.hasComment(commentTitle));
    assert.ok(comment);
  });
});

test('can remove a comment', function (assert) {
  assert.expect(4);
  stubLogin({ username: channelName });
  this.server.create('post');

  let commentTitle = 'Best comment ever';

  visit(page.url());
  fillIn(page.commentTextarea, commentTitle);
  keyEvent(page.commentTextarea, 'keydown', 13);

  andThen(() => {
    assert.equal(this.server.db.comments.length, 1, 'The comment has been created in the DB.');
    assert.ok(page.hasComment(commentTitle));
  });

  click(page.commentModControls);
  click(page.deleteCommentMenu);
  click(page.confirmDeleteComment);

  andThen(() => {
    assert.equal(this.server.db.comments.length, 0, 'The comment has been deleted from the DB.');
    assert.notOk(page.hasComment(commentTitle), 'Comment removed from template');
  });
});
