import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';
import subscriptionDashScenario from 'web-client/mirage/scenarios/subscription-dash';
import { AdsPage } from 'web-client/tests/pages/dashboards/subscription-dash';
import { stubLogin } from 'web-client/tests/helpers/stub-login';

const name = 'twitch';

moduleForAcceptance('Acceptance | dashboard/subscription/ads', {
  beforeEach() {
    this.setupServer = (options) => {
      subscriptionDashScenario(this.server, options);
      this.page = AdsPage.create({ username: options.name });
    };

    stubLogin({ login: name });
  }
});

test('visits /dashboard/subscription/ads and goes back to /dashboard/subscription index page', function (assert) {
  assert.expect(4);

  this.setupServer({ name });
  visit(this.page.url());

  andThen(() => {
    assert.equal(
      currentURL(),
      `/twitch/dashboard/subscription/ads`,
      `/twitch/dashboard/subscription/ads is visited`
    );

    assert.equal(
      find(this.page.pageTitle()).text().trim(),
      'Ad-Free Viewing for Subscribers',
      'title of "Ad-Free Viewing for Subscribers" is present'
    );
  });

  click(this.page.backToIndex());

  andThen(() => {
    assert.equal(
      currentURL(),
      `/twitch/dashboard/subscription`,
      `/twitch/dashboard/subscription is visited`
    );

    assert.equal(this.page.headTitle(), `${name}'s Dashboard - Twitch`, 'has title tag in head');
  });
});

test('"Ad-Free Viewing for Subscribers" is unchecked when NOT hiding ads', function (assert) {
  assert.expect(1);

  this.setupServer({ hideAds: false, name });
  visit(this.page.url());

  andThen(() => {
    assert.notOk(
      find(this.page.checkbox()).prop('checked'),
      'checkbox is unchecked when NOT ads-free for subscribers'
    );
  });
});

test('"Ad-Free Viewing for Subscribers" is checked when hiding ads', function (assert) {
  assert.expect(1);

  this.setupServer({ hideAds: true, name });
  visit(this.page.url());

  andThen(() => {
    assert.ok(
      find(this.page.checkbox()).prop('checked'),
      'checkbox is checked when ads-free for subscribers'
    );
  });
});

test('opens warning modal after unchecking "Ad-Free Viewing for Subscribers"', function (assert) {
  assert.expect(2);

  this.setupServer({ hideAds: true, name });
  visit(this.page.url());

  andThen(() => {
    assert.ok(
      find(this.page.checkbox()).prop('checked'),
      'precond - checkbox is checked when ads-free for subscribers'
    );
  });

  click(this.page.checkbox());

  andThen(() => {
    assert.equal(
      find(this.page.warningModal()).text().trim(),
      'Warning: Viewers expect consistency and changing this feature frequently could negatively impact your community.',
      'warning message is present'
    );
  });
});

test('checkbox unchecks after confirming warning modal', function (assert) {
  assert.expect(3);

  this.setupServer({ hideAds: true, name });
  visit(this.page.url());

  andThen(() => {
    assert.ok(
      find(this.page.checkbox()).prop('checked'),
      'precond - checkbox is checked when ads-free for subscribers'
    );
  });

  click(this.page.checkbox());

  andThen(() => {
    assert.equal(
      find(this.page.warningModal()).text().trim(),
      'Warning: Viewers expect consistency and changing this feature frequently could negatively impact your community.',
      'precond - warning message is present'
    );
  });

  click(this.page.warningModalConfirmButton());

  andThen(() => {
    assert.notOk(
      find(this.page.checkbox()).prop('checked'),
      'checkbox is unchecked after confirming warning'
    );
  });
});

test('checkbox remains checked after cancelling warning modal', function (assert) {
  assert.expect(3);

  this.setupServer({ hideAds: true, name });
  visit(this.page.url());

  andThen(() => {
    assert.ok(
      find(this.page.checkbox()).prop('checked'),
      'precond - checkbox is checked when ads-free for subscribers'
    );
  });

  click(this.page.checkbox());

  andThen(() => {
    assert.equal(
      find(this.page.warningModal()).text().trim(),
      'Warning: Viewers expect consistency and changing this feature frequently could negatively impact your community.',
      'precond - warning message is present'
    );
  });

  click(this.page.warningModalCloseButton());

  andThen(() => {
    assert.ok(
      find(this.page.checkbox()).prop('checked'),
      'checkbox remains checked after cancelling warning'
    );
  });
});
