import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';
import followScenario from 'web-client/mirage/scenarios/follow';
import channelScenario from 'web-client/mirage/scenarios/channel';
import AllPage from 'web-client/tests/pages/directory/all';
import { stubLogin } from 'web-client/tests/helpers/stub-login';

moduleForAcceptance('Acceptance | directory/all', {
  beforeEach() {
    this.experiments.use({ WATCH_NOW_RIGHTBAR: 'control' });
    this.page = AllPage.create();
    stubLogin({ login: 'twitch' });
    this.server.create('channel-ember-data', { name: 'twitch' });
  }
});

test('logged in users sees no follows message', function(assert) {
  assert.expect(4);

  visit(this.page.url());

  andThen(() => {
    assert.ok(this.page.youFollowTitle(), 'You Follow Title on the page');
    assert.equal(this.page.headTitle(), 'Top Channels - Twitch', 'has title tag in head');
    assert.ok(this.page.noFollows(), 'No follows message');
    assert.equal(this.page.canonicalUrl(), 'https://www.twitch.tv/directory/all', 'Canonical URL present');
  });
});

test("/directory/all tracks page latency by reporting when it's ready to accept user interaction", function(assert) {
  assert.expect(2);
  visit(this.page.url());

  andThen(() => {
    assert.trackInteractivity('directory.channels.all');
  });

  andThen(() => {
    assert.trackBenchmarkEvent('complete_transition');
  });
});

test("/directory/all/ps4 tracks page latency by reporting when it's ready to accept user interaction", function(assert) {
  assert.expect(2);
  visit(this.page.psFourURL);

  andThen(() => {
    assert.trackInteractivity('directory.channels.psFour');
  });

  andThen(() => {
    assert.trackBenchmarkEvent('complete_transition');
  });
});

test("/directory/all/xb1 tracks page latency by reporting when it's ready to accept user interaction", function(assert) {
  assert.expect(2);
  visit(this.page.xbOneURL);

  andThen(() => {
    assert.trackInteractivity('directory.channels.xbOne');
  });

  andThen(() => {
    assert.trackBenchmarkEvent('complete_transition');
  });
});

test('logged in users with offline follows can click to open channel videos list', function(assert) {
  assert.expect(5);

  channelScenario(this.server, 'counterpit');
  let follow = followScenario(this.server, { isOffline: true });
  let offlineUrl = this.page.offlineUrl();

  visit(this.page.url());

  andThen(() => {
    assert.ok(this.page.youFollowTitle(), 'You Follow title on the page');

    let link = this.page.followedChannelLink('offline');
    assert.ok(link, 'Followed channel link found');
    assert.equal(link.attr('href'), offlineUrl, `offline profile href on link`);

    assert.ok(this.page.followImageLink('offline', offlineUrl, follow.channel.logo), 'clickable image');
    click(link);
  });

  andThen(() => {
    assert.equal(currentURL(), this.page.allVideosUrl(), 'transitioned to followed user videos list');
  });
});

test('logged in users with online follows can click to open channel', function(assert) {
  assert.expect(5);

  let follow = followScenario(this.server);

  visit(this.page.url());

  andThen(() => {
    assert.ok(this.page.youFollowTitle(), 'You Follow title on the page');

    let link = this.page.followedChannelLink('online');
    assert.ok(link, 'Followed channel link found');
    assert.equal(link.attr('href'), this.page.fullOnlineUrl(), 'online channel href on link');

    assert.ok(this.page.liveThumbLink(follow.channelName), 'follow thumbnail in main content of directory page');

    click(link);
  });

  andThen(() => {
    assert.equal(currentURL(), this.page.onlineUrl(), 'transitioned to channel page');
  });
});

test('logged in users with playlist follows can click to open channel', function(assert) {
  assert.expect(5);

  let follow = followScenario(this.server, {
    isOffline: false,
    isPlaylist: true
  });

  visit(this.page.url());

  andThen(() => {
    assert.ok(this.page.youFollowTitle(), 'You Follow title on the page');

    let link = this.page.followedChannelLink('playlist');
    assert.ok(link, 'Followed channel link found');
    assert.equal(link.attr('href'), this.page.fullOnlineUrl(), 'online channel href on link');

    assert.ok(this.page.liveThumbLink(follow.channelName), 'follow thumbnail in main content of directory page');

    click(link);
  });

  andThen(() => {
    assert.equal(currentURL(), this.page.onlineUrl(), 'transitioned to channel page');
  });
});

test('following filtering', function(assert) {
  assert.expect(5);

  // 2 offline follows
  followScenario(this.server, { isOffline: true, name: 'zzaaoff' });
  followScenario(this.server, { isOffline: true, name: 'zzbboff' });

  // 2 online follows
  followScenario(this.server, { name: 'zzaaon' });
  followScenario(this.server, { name: 'zzbbon' });

  // 2 playlist follows
  followScenario(this.server, { isPlaylist: true, name: 'zzaapl' });
  followScenario(this.server, { isPlaylist: true, name: 'zzbbpl' });

  visit(this.page.url());

  andThen(() => {
    assert.equal(this.page.channels().length, 6, 'follows on page with no filtering');
  });

  click(this.page.searchButton());
  fillIn(this.page.input(), 'aa');

  andThen(() => {
    assert.equal(this.page.channels().length, 3, 'follows filtered to 3');
    assert.ok(this.page.followedChannelLink('offline', 'zzaaoff'), 'filtered offline');
    assert.ok(this.page.followedChannelLink('online', 'zzaaon'), 'filtered online');
    assert.ok(this.page.followedChannelLink('playlist', 'zzaapl'), 'filtered offline');
  });
});

test('following filtering by game', function(assert) {
  assert.expect(5);

  // 2 offline follows
  followScenario(this.server, { isOffline: true, name: 'c1', game: 'zzaaoff' });
  followScenario(this.server, { isOffline: true, name: 'c2', game: 'zzbboff' });

  // 2 online follows
  followScenario(this.server, { name: 'c3', game: 'zzaaon' });
  followScenario(this.server, { name: 'c4', game: 'zzbbon' });

  // 2 playlist follows
  followScenario(this.server, { isPlaylist: true, name: 'c5', game: 'zzaapl' });
  followScenario(this.server, { isPlaylist: true, name: 'c6', game: 'zzbbpl' });

  visit(this.page.url());

  andThen(() => {
    assert.equal(this.page.channels().length, 6, 'follows on page with no filtering');
  });

  click(this.page.searchButton());
  fillIn(this.page.input(), 'aa');

  andThen(() => {
    assert.equal(this.page.channels().length, 3, 'follows filtered to 3');
    assert.ok(this.page.followedChannelLink('offline', 'c1'), 'filtered offline');
    assert.ok(this.page.followedChannelLink('online', 'c3'), 'filtered online');
    assert.ok(this.page.followedChannelLink('playlist', 'c5'), 'filtered offline');
  });
});

test('follow pagination in sidebar', function(assert) {
  assert.expect(2);

  for (let i = 0; i < 60; i++) {
    followScenario(this.server, { isOffline: true, name: `follow${i}` });
  }

  visit(this.page.url());

  andThen(() => {
    assert.equal(this.page.followedChannelLink('offline', null).length, 25, 'first page of 25 follows');

    click(this.page.loadMore);
  });

  andThen(() => {
    assert.equal(this.page.followedChannelLink('offline', null).length, 50, 'two pages showing 50 follows');
  });
});

test('live following directory; pagination', function(assert) {
  assert.expect(3);

  for (let i = 0; i < 60; i++) {
    followScenario(this.server, { name: `follow${i}`});
  }

  visit(this.page.liveUrl);

  andThen(() => {
    assert.equal(currentURL(), this.page.liveUrl, `loaded ${this.page.liveUrl}`);
    assert.equal(find(this.page.streamItem).length, 24, 'initial page of 24 loaded');

    this.page.scrollDown();
  });

  andThen(() => {
    assert.equal(find(this.page.streamItem).length, 48, 'second page loaded, showing 48 total');
  });
});
