import { test } from 'qunit';
import moduleForAcceptance from '../../helpers/module-for-acceptance';
import DirectoryDiscoverPage from '../../pages/directory/discover';
import { stubLogin } from 'web-client/tests/helpers/stub-login';
import MockRecsHoldout from 'web-client/tests/helpers/mock-recs-holdout-service';
import recommendedStreamsScenario from 'web-client/mirage/scenarios/recommended-streams';
import recentlyWatchedStreamsScenario from 'web-client/mirage/scenarios/recently-watched-streams';
import trendingVideosScenario from 'web-client/mirage/scenarios/trending-videos';

let page;

moduleForAcceptance('Acceptance | /directory/discover', {
  beforeEach() {
    page = DirectoryDiscoverPage.create();
    stubLogin({ login: 'twitch', chat_oauth_token: 'foobar' });
  }
});

test('visiting /directory/discover page without recommendations available', function (assert) {
  assert.expect(5);

  recommendedStreamsScenario(this.server, {available: false});
  recentlyWatchedStreamsScenario(this.server, {streamsCount: 0});
  trendingVideosScenario(this.server, {videosCount: 0});

  visit(page.url());

  andThen(function () {
    assert.equal(currentURL(), page.url());
    assert.equal(currentRouteName(), 'directory.discover');
    assert.equal(find(page.emptyMessage()).length, 1, 'has empty message');
    assert.trackInteractivity('directory.discover');
  });

  andThen(() => { // Run in a separate andThen() because this gets fired from a resolved promise
    assert.trackBenchmarkEvent('complete_transition');
  });
});

test('visiting /directory/discover page with no recommendations', function (assert) {
  assert.expect(5);

  recommendedStreamsScenario(this.server, {available: true});
  recentlyWatchedStreamsScenario(this.server, {streamsCount: 0});
  trendingVideosScenario(this.server, {videosCount: 0});

  visit(page.url());

  andThen(function () {
    assert.equal(currentURL(), page.url());
    assert.equal(currentRouteName(), 'directory.discover');
    assert.equal(find(page.emptyMessage()).length, 1, 'has empty message');
    assert.trackInteractivity('directory.discover');
  });

  andThen(() => { // Run in a separate andThen() because this gets fired from a resolved promise
    assert.trackBenchmarkEvent('complete_transition');
  });
});

test('visiting /directory/discover page with recommendations with carousel', function (assert) {
  assert.expect(6);

  this.owner.register('service:recs-holdout', MockRecsHoldout);

  recommendedStreamsScenario(this.server, {available: true, channelsCount: 1, streamsCount: 2});
  recentlyWatchedStreamsScenario(this.server, {streamsCount: 0});
  trendingVideosScenario(this.server, {videosCount: 0});

  visit(page.url());

  andThen(function () {
    assert.equal(currentURL(), page.url());
    assert.equal(currentRouteName(), 'directory.discover');
    assert.equal(find(page.emptyMessage()).length, 0, 'no empty message');
    assert.equal(find(page.carouselItems()).length, 2, 'has right number of carousel items');
    assert.trackInteractivity('directory.discover');
  });

  andThen(() => { // Run in a separate andThen() because this gets fired from a resolved promise
    assert.trackBenchmarkEvent('complete_transition');
  });
});

test('dismissing help', function(assert) {
  assert.expect(2);

  recommendedStreamsScenario(this.server, {available: true});
  recentlyWatchedStreamsScenario(this.server, {streamsCount: 0});
  trendingVideosScenario(this.server, {videosCount: 0});

  visit(page.url());

  andThen(() => {
    assert.equal(find(page.help()).length, 1, "shows the help");
  });

  click(page.dismissHelp());

  andThen(() => {
    assert.equal(find(page.help()).length, 0, "help section is gone");
  });
});
