import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';
import FollowingVideosPage from 'web-client/tests/pages/directory/following-videos';
import { followingLoggedOutMessage } from 'web-client/tests/pages/directory/following';
import testForLoginModal from 'web-client/tests/helpers/test-for-login-modal';
import { stubLogin } from 'web-client/tests/helpers/stub-login';
import videoAppealScenario from 'web-client/mirage/scenarios/video-appeal';
import channelScenario from 'web-client/mirage/scenarios/channel';
import MockRecsHoldout from 'web-client/tests/helpers/mock-recs-holdout-service';
import { stubMixpanelId } from 'web-client/tests/helpers/mock-ids-for-mixpanel';

moduleForAcceptance('Acceptance | directory/following/videos [logged out]', {
  beforeEach() {
    this.page = FollowingVideosPage.create();
  }
});

testForLoginModal({
  url: '/directory/following/videos',
  testFunction: test,
  expectedText: followingLoggedOutMessage
});

test('tracks interactivity [logged out]', function (assert) {
  assert.expect(2);

  visit(this.page.url());

  andThen(function () {
    assert.trackInteractivity('directory.following.videos.video-type');
  });

  andThen(() => { // Run in a separate andThen() because this gets fired from a resolved promise
    assert.trackBenchmarkEvent('complete_transition');
  });
});

moduleForAcceptance('Acceptance | directory/following/videos [logged in]', {
  beforeEach() {
    this.page = FollowingVideosPage.create();
    stubLogin();

    // Necessary to ensure vod_appeals API returns data on the vod page
    videoAppealScenario(this.server, 1);

    this.channel = channelScenario(this.server);
    this.video = this.server.create('video', { channel: this.channel });
  }
});

test('clicking video sends "follow_directory_click" tracking event', function(assert) {
  assert.expect(2);

  // Create a video-follow object so that the video shows up in the list of
  // followed videos
  this.server.create('video-follow', { videoId: this.video.id });

  let videoIndex = 0;

  visit(this.page.url());
  click(this.page.videoAtIndex(videoIndex));

  andThen(() => {
    assert.equal(currentRouteName(), 'videos');
    assert.equal(currentURL(), `/videos/${this.video.id}`);
  });
});

test('tracks interactivity - no recs', function (assert) {
  assert.expect(2);

  this.server.create('video-follow', { videoId: this.video.id });

  visit(this.page.url());

  andThen(function () {
    assert.trackInteractivity('directory.following.videos.video-type');
  });

  andThen(() => { // Run in a separate andThen() because this gets fired from a resolved promise
    assert.trackBenchmarkEvent('complete_transition');
  });
});

test('clicking recommended video sends "coview_recommendation_click" tracking event', function(assert) {
  assert.expect(3);

  this.owner.register('service:recs-holdout', MockRecsHoldout);

  let videoIndex = 0;
  let mixpanelId = 'random-mixpanel-id';
  stubMixpanelId(mixpanelId);

  let expectedEventData = {
    game: undefined,
    channel: this.channel.name,
    num_recs_rendered: 1,
    rank_position: videoIndex,
    vod_id: `v${this.video.id}`
  };

  visit(this.page.url());
  click(this.page.recommendedVideoAtIndex(videoIndex));

  andThen(() => {
    assert.trackEvent('coview_recommendation_click', expectedEventData);
    assert.equal(currentRouteName(), 'videos');
    assert.equal(currentURL(), `/videos/${this.video.id}`);
  });
});

test('tracks interactivity - w/ recs', function (assert) {
  assert.expect(2);

  this.owner.register('service:recs-holdout', MockRecsHoldout);
  this.server.create('video-follow', { videoId: this.video.id });

  visit(this.page.url());

  andThen(function () {
    assert.trackInteractivity('directory.following.videos.video-type');
  });

  andThen(() => { // Run in a separate andThen() because this gets fired from a resolved promise
    assert.trackBenchmarkEvent('complete_transition');
  });
});
