import { test } from 'qunit';
import moduleForAcceptance from '../../helpers/module-for-acceptance';
import channelDirectoryScenario from 'web-client/mirage/scenarios/channel-directory';
import DirectoryGamePage from '../../pages/directory/game';
import DirectoryFollowingPage from '../../pages/directory/following';
import DirectoryFollowingGamesPage from '../../pages/directory/following-games';
import { stubLogin } from '../../helpers/stub-login';

moduleForAcceptance('Acceptance | directory/game', {
  beforeEach() {
    channelDirectoryScenario(this.server);
    /* lazy-create the page for each test after its game is created */
    Object.defineProperty(this, 'page', {
      get() {
        return DirectoryGamePage.create({name: this.game.name});
      }
    });
  }
});

test('visiting directory game page when game has slash character', function (assert) {
  assert.expect(4);

  let name = 'AC/DC';
  this.game = this.server.create('game', {name});

  visit(this.page.url());

  assert.stringIncludes(this.game.name, '/', `precond - game name "${this.game.name}" has slash`);

  andThen(() => {
    assert.equal(currentURL(), this.page.url());
    assert.equal(currentRouteName(), 'directory.game.index');
    assert.equal(find(this.page.title()).length, 1, 'has Title');
  });
});

test("tracks page latency by reporting when it's ready to accept user interaction", function(assert) {
  assert.expect(2);

  this.game = this.server.create('game', {
    name: 'Minecraft 2'
  });

  visit(this.page.url());

  andThen(() => {
    assert.trackInteractivity('directory.game.index');
  });

  // Run in a separate andThen() because this gets fired from a resolved promise
  andThen(() => {
    assert.trackBenchmarkEvent('complete_transition');
  });
});

test('language directory filter appears when in control group', function(assert) {
  this.experiments.use({'LANGUAGE_DIRECTORY_FILTER': 'control'});

  /* The default game is Creative, here we want an arbitrary non-Creative game */
  this.game = this.server.create('game', {name: 'Random Non-Creative Game'});

  assert.expect(1);

  visit(this.page.url());

  andThen(() => {
    assert.equal(find(this.page.languageFilter()).length, 1, 'language filter is not visible');
  });
});

test('language directory filter appears when in experiment group', function(assert) {
  assert.expect(2);

  this.experiments.use({'LANGUAGE_DIRECTORY_FILTER': 'full'});

  /* The default game is Creative, here we want an arbitrary non-Creative game */
  this.game = this.server.create('game', {name: 'Random Non-Creative Game'});

  visit(this.page.url());

  andThen(() => {
    assert.equal(find(this.page.languageFilter()).length, 1, 'language filter is visible');
  });

  this.experiments.use({'LANGUAGE_DIRECTORY_FILTER': 'scroll'});

  visit(this.page.url());

  andThen(() => {
    assert.equal(find(this.page.languageFilter()).length, 1, 'language filter is visible');
  });
});

test(`visiting lowercase version of game name works [WEBP-60]`, function (assert) {
  assert.expect(3);

  let name = 'Destiny';
  this.game = this.server.create('game', {name});

  visit(this.page.url().toLowerCase());

  andThen(() => {
    assert.equal(currentURL(), this.page.url(), 'properly redirected');
    assert.equal(currentRouteName(), 'directory.game.index');
    assert.equal(find(this.page.title(name)).length, 1, 'has Title');
  });
});

test(`visiting non-existing directory game page`, function (assert) {
  assert.expect(1);

  visit(`/directory/game/not-found`);

  andThen(() => {
    assert.equal(find('.ember_error_404').length, 1, "renders the 404 page");
  });
});


test(`visiting game page and following shows game on following overview and following games pages`, function (assert) {
  assert.expect(4);
  stubLogin();

  this.game = this.server.create('game', {name: 'Random Non-Creative Game'});

  visit(this.page.url());
  andThen(() => {
    assert.ok(this.page.isNotFollowing(), 'precond - user is not following game');
  });
  click(this.page.followButtonSelector);
  andThen(() => {
    assert.ok(this.page.isFollowing(), 'user is following game');
  });
  click(this.page.sidebarFollowingLinkSelector);
  andThen(() => {
    let followingPage = DirectoryFollowingPage.create();

    assert.ok(followingPage.hasGameIconFor(this.game.name), `Following page has game icon for ${this.game.name}`);

    click(followingPage.gameTabSelector);
  });
  andThen(() => {
    let followingGamesPage = DirectoryFollowingGamesPage.create();

    assert.ok(followingGamesPage.hasGameIconFor(this.game.name), `Following games page has game icon for ${this.game.name}`);
  });
});

test(`visiting game page and unfollowing removes game from following overview and following games pages`, function (assert) {
  assert.expect(4);
  stubLogin();

  this.game = this.server.create('game', {name: 'Random Non-Creative Game'});
  this.server.create('game-follow', {gameId: this.game.id});

  visit(this.page.url());
  andThen(() => {
    assert.ok(this.page.isFollowing(), 'precond - user is following game');
  });
  click(this.page.unfollowButtonSelector);
  andThen(() => {
    assert.ok(this.page.isNotFollowing(), 'user is not following game');
  });
  click(this.page.sidebarFollowingLinkSelector);
  andThen(() => {
    let followingPage = DirectoryFollowingPage.create();

    assert.notOk(followingPage.hasGameIconFor(this.game.name), `Following page has no game icon for ${this.game.name}`);

    click(followingPage.gameTabSelector);
  });
  andThen(() => {
    let followingGamesPage = DirectoryFollowingGamesPage.create();

    assert.notOk(followingGamesPage.hasGameIconFor(this.game.name), `Following games page has no game icon for ${this.game.name}`);
  });
});

moduleForAcceptance('Acceptance | empty directory/game', {
  beforeEach() {
    /* lazy-create the page for each test after its game is created */
    Object.defineProperty(this, 'page', {
      get() {
        return DirectoryGamePage.create({name: this.game.name});
      }
    });
  }
});

test("tracks page latency by reporting when it's ready to accept user interaction", function(assert) {
  assert.expect(2);

  this.game = this.server.create('game', {
    name: 'Some rly unpopular game'
  });

  visit(this.page.url());

  andThen(() => {
    assert.trackInteractivity('directory.game.index');
  });

  // Run in a separate andThen() because this gets fired from a resolved promise
  andThen(() => {
    assert.trackBenchmarkEvent('complete_transition');
  });
});
