import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';
import videoScenario from 'web-client/mirage/scenarios/video';
import followScenario from 'web-client/mirage/scenarios/follow';
import VideosDirectoryPage from 'web-client/tests/pages/directory/videos';
import { stubLogin } from 'web-client/tests/helpers/stub-login';

moduleForAcceptance('Acceptance | directory/videos', {
  beforeEach() {
    this.page = VideosDirectoryPage.create();
    this.video = videoScenario(this.server, { channelName: 'rwjblue', views: 5 });
    videoScenario(this.server, { channelName: 'foo-bar', views: 5 });
    videoScenario(this.server, { channelName: 'bar-foo', views: 5 });
  }
});

test('visiting videos directory page', function(assert) {
  assert.expect(6);

  visit(this.page.url());

  andThen(() => {
    assert.equal(currentURL(), '/directory/videos/all');
    assert.equal(currentRouteName(), 'directory.videos.video-type');

    assert.equal(find(this.page.videosTitle()).length, 1, 'has videos Title');
    assert.equal(this.page.headTitle(), 'All Videos - Twitch', 'has title tag in head');
    assert.equal(find(this.page.directoryGridVideoThumbs()).length, 3, 'has 3 video thumbs');
    assert.notOk(this.page.hasProgressBar(), 'non-resumetime videos have no progress bar');
  });
});

test('visiting videos directory page', function(assert) {
  assert.expect(2);

  visit(this.page.url());

  andThen(() => {
    assert.trackInteractivity('directory.videos.video-type');
  });

  andThen(() => {
    assert.trackBenchmarkEvent('complete_transition');
  });
});

test('visiting videos directory page with show trending experiment (insufficient uploads)', function(assert) {
  assert.expect(3);

  visit(this.page.url());

  andThen(() => {
    assert.equal(currentURL(), '/directory/videos/all');
    assert.equal(currentRouteName(), 'directory.videos.video-type');

    assert.equal(find(this.page.carouselVideos()).length, 0, 'has 0 uploads in carousel');
  });
});

test('visiting videos directory page with show trending experiment (insufficient views)', function(assert) {
  assert.expect(3);

  videoScenario(this.server, { channelName: 'foo-bar', views: 5 });
  videoScenario(this.server, { channelName: 'bar-foo', views: 4 });

  visit(this.page.url());

  andThen(() => {
    assert.equal(currentURL(), '/directory/videos/all');
    assert.equal(currentRouteName(), 'directory.videos.video-type');

    assert.equal(find(this.page.carouselVideos()).length, 0, 'has 0 uploads in carousel');
  });
});

test('visiting videos directory page with show trending experiment', function(assert) {
  assert.expect(3);

  videoScenario(this.server, { channelName: 'foo-bar', views: 5 });
  videoScenario(this.server, { channelName: 'bar-foo', views: 5 });

  visit(this.page.url());

  andThen(() => {
    assert.equal(currentURL(), '/directory/videos/all');
    assert.equal(currentRouteName(), 'directory.videos.video-type');

    assert.equal(find(this.page.carouselVideos()).length, 5, 'has 5 uploads in carousel');
  });
});

test('clicking video thumbnail takes you to video page', function(assert) {
  assert.expect(2);

  visit(this.page.url());

  click(this.page.thumbLink(this.video.url));

  andThen(() => {
    assert.equal(currentURL(), `/videos/${this.video.id}`, 'transitioned');
    assert.equal(currentRouteName(), 'videos');
  });
});

test('a watched video has a progress bar for logged in users', function(assert) {
  assert.expect(1);
  let username = 'twitch';
  let userId = 10000000;

  stubLogin({ login: username, id: userId });
  //TODO: this scenario only required to load the follow column, can be removed after graceful retry cleanup
  followScenario(this.server, { name: username });

  let videoId = this.video._id;

  this.server.get('https://api.twitch.tv/api/resumewatching/user', function() {
    return {
      videos: [ { video_id: videoId, position: 30 } ]
    };
  });

  visit(this.page.url());

  andThen(() => {
    assert.ok(this.page.hasProgressBar());
  });
});
