import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';
import { stubLogin } from 'web-client/tests/helpers/stub-login';
import PageObject from 'web-client/tests/pages/friends-list';
import friendsListScenario from 'web-client/mirage/scenarios/friends-list';
import channelScenario from 'web-client/mirage/scenarios/channel';

moduleForAcceptance('Acceptance | friends-list open requests list', {
  beforeEach() {
    let channel = channelScenario(this.server, 'twitch');
    this.currentUser = stubLogin({ login: channel.name, id: channel._id });
    friendsListScenario(this.server);
    this.page = PageObject.create();
  }
});

test('clicking on the Requests tab → /friends/requests?limit=21', function(assert) {
  assert.expect(1);
  this.server.get('https://api.twitch.tv/kraken/users/:user_id/friends/requests', (schema, request) => {
    assert.equal(request.queryParams.limit, '21', 'passes ?limit=21');
    return {requests: []};
  });

  visit(this.page.url());
  click(this.page.openFriendRequestButton);
});


test('opening the requests with notifications present → DELETE /friends/notifications', function(assert) {
  assert.expect(1);

  this.server.del('https://api.twitch.tv/kraken/users/:user_id/friends/notifications', () => {
    assert.ok(true, 'cleared notifications');
    return {requests: []};
  });

  visit(this.page.url());
  click(this.page.openFriendRequestButton);
});

test('over 20 unseen invites will display 20+', function(assert) {
  assert.expect(1);

  this.server.get('https://api.twitch.tv/kraken/users/:user_id/friends/notifications', { count: 21 });

  visit(this.page.url());

  andThen(() => {
    let count = find(this.page.unseenRequestCount).text();
    assert.ok(/20\+/.test(count), 'div contains desired count of 20+');
  });
});

test('accepting a friend → PUT /:friend_id', function(assert) {
  assert.expect(2);
  this.server.create('friend-request', { targetId: this.currentUser.id, senderId: '12345' });
  this.server.put('https://api.twitch.tv/kraken/users/:user_id/friends/:id', () => {
    assert.ok(true, 'Made request successfully');
    return [204, {}, ''];
  });

  visit(this.page.url());
  click(this.page.openFriendRequestButton);
  click(`${this.page.requestActions}:eq(0)`);
  andThen(() => {
    assert.notOk(find(this.page.requestActions).length, 'Request disappeared');
  });
});

test('rejecting a friend → DELETE /:friend_id', function(assert) {
  assert.expect(1);
  this.server.create('friend-request', {targetId: this.currentUser.id, senderId: '12345'});

  this.server.del('https://api.twitch.tv/kraken/users/:user_id/friends/requests/:id', () => {
    assert.ok(true, 'Made request successfully');
    return {};
  });

  visit(this.page.url());

  click(this.page.openFriendRequestButton);
  click(`${this.page.requestActions}:eq(1)`);
});
