import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';
import HighlightsPage from 'web-client/tests/pages/video-manager/highlights';
import { stubLogin } from 'web-client/tests/helpers/stub-login';
import { HIGHLIGHT_TYPE_PARAM } from 'web-client/utilities/video/types';

const USERNAME = 'mitchlloyd';
const GENERATED_THUMBNAIL_SRC = '/fixtures/images/stream/2/320x180.jpg';
const CUSTOM_THUMBNAIL_SRC = '/fixtures/images/stream/1/320x180.jpg';
const NUM_VIEWS = 101;
const DURATION = 128;
const TITLE = "My Highlight";
const DESCRIPTION = "my description";
const TAGS = 'fun,and,games';

moduleForAcceptance('Acceptance | /manager/highlights', {
  beforeEach() {
    this.channel = this.server.create('channel', { name: USERNAME });
    this.page = HighlightsPage.create({ USERNAME });
    stubLogin({ login: USERNAME });
    this.video = {
      broadcast_type: HIGHLIGHT_TYPE_PARAM,
      title: TITLE,
      description: DESCRIPTION,
      descriptionHTML: DESCRIPTION,
      descriptionMarkdown: DESCRIPTION,
      tag_list: 'fun,and,games',
      length: DURATION,
      created_at: '2017-01-27T12:14:53Z',
      published_at: null,
      thumbnails: {
        small: [],
        medium: [
          {type: 'generated', url: GENERATED_THUMBNAIL_SRC},
          {type: 'custom', url: CUSTOM_THUMBNAIL_SRC}
        ],
        large: [],
        template: []
      },
      preview: {
        small: '/fixtures/images/stream/1/52x72.jpg',
        medium: '/fixtures/images/stream/1/320x180.jpg'
      },
      views: NUM_VIEWS,
      channel: this.channel
    };
  }
});

test('visiting /:username/manager/highlights with a highlight', function(assert) {
  assert.expect(2);

  this.server.create('video', this.video);

  visit(this.page.url());

  andThen(() => {
    assert.equal(currentURL(), this.page.url());
    let videoElement = find('.card__title:contains(My Highlight)');
    assert.equal(videoElement.length, 1, "page shows the highlight");
  });
});

test('visiting /:username/manager/highlights without a highlight', function(assert) {
  assert.expect(1);
  visit(this.page.url());

  andThen(() => {
    assert.ok(this.page.hasEmptyMessage(), "page shows empty message");
  });
});

test('visiting /:username/manager/highlights and edit video', function(assert) {
  assert.expect(14);

  this.server.create('video', this.video);

  visit(this.page.url());
  andThen(() => {
    assert.equal(this.page.videoTitle(), TITLE, 'precond - correct video title');
    assert.equal(this.page.videoDescription(), DESCRIPTION, 'precond - correct video desc');
    assert.ok(this.page.videoInfo().indexOf("Jan 27, 2017") >= 0, 'date is correctly set');
    assert.ok(this.page.videoInfo().indexOf("2:08") >= 0, 'duration is correctly set');
  });
  this.page.clickEdit();
  andThen(() => {
    assert.equal(this.page.editTitleVal(), TITLE, 'edit modal has correct title');
    assert.equal(this.page.editDescriptionVal(), DESCRIPTION, 'edit modal has correct description');
    assert.equal(this.page.editTagsVal(), TAGS, 'edit modal has correct tags');
    assert.equal(this.page.mainThumbnailSrc(), CUSTOM_THUMBNAIL_SRC, 'custom thumbnail is selected');
    assert.equal(this.page.thumbnails().length, 2, 'edit modal has correct number of thumbnails');
  });
  this.page.editVideo({
    title: 'new title',
    descriptionMarkdown: 'new desc',
    tags: 'tag1,tag2,tag3'
  });
  andThen(() => {
    assert.ok(this.page.isFound(this.page.editVideoModalSelector), 'shows edit video modal');
  });
  click(this.page.editVideoDoneButtonSelector);
  andThen(() => {
    let video = this.server.schema.videos.first();
    assert.equal(video.title, 'new title');
    assert.equal(video.description, 'new desc');
    assert.equal(video.tag_list, 'tag1,tag2,tag3');

    assert.equal(this.page.videoTitle(), 'new title');
  });
});

test('visiting /:username/manager/highlights has expected options on video card', function(assert) {
  assert.expect(7);

  this.server.create('video', this.video);

  visit(this.page.url());
  andThen(() => {
    let editButton = find('[data-test-selector=edit-button]');
    let moreMenu = find('[data-test-selector=more-menu]');
    let moreButton = moreMenu.find('[data-test-selector=more-menu-button]');
    let downloadButton = moreMenu.find('a:contains(Download)');
    let deleteButton = moreMenu.find('[data-test-selector=delete-button]');
    let exportButton = moreMenu.find('[data-test-selector=export-button]');
    let videoStatsButton = moreMenu.find('[data-test-selector=video-stats-button]');

    assert.equal(editButton.length, 1, 'Edit button present');
    assert.equal(moreMenu.length, 1, 'More menu present');
    assert.equal(moreButton.length, 1, 'More button present');
    assert.equal(downloadButton.length, 1, 'More menu should have download button');
    assert.equal(deleteButton.length, 1, 'More menu should have delete button');
    assert.equal(exportButton.length, 1, 'More menu should have export button');
    assert.equal(videoStatsButton.length, 1, 'More menu should have video stats button');
  });
});
