import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';
import channelScenario from 'web-client/mirage/scenarios/channel';
import videoAppealScenario from 'web-client/mirage/scenarios/video-appeal';
import PastBroadcastsPage from 'web-client/tests/pages/video-manager/past-broadcasts';
import { stubLogin } from 'web-client/tests/helpers/stub-login';
import testForLoginModal from 'web-client/tests/helpers/test-for-login-modal';
import { ARCHIVE_TYPE_PARAM } from 'web-client/utilities/video/types';

moduleForAcceptance('Acceptance | /manager/past-broadcasts (logged out)');

testForLoginModal({
  url: '/logged-out-user/manager/past_broadcasts',
  testFunction: test
});

const USERNAME = 'mitchlloyd';
const GENERATED_THUMBNAIL_SRC = '/fixtures/images/stream/2/320x180.jpg';
const CUSTOM_THUMBNAIL_SRC = '/fixtures/images/stream/1/320x180.jpg';
const NUM_VIEWS = 101;
const DURATION = 128;
const TITLE = "My Broadcast";
const DESCRIPTION = "my description";
const TAGS = 'fun,and,games';

moduleForAcceptance('Acceptance | /manager/past-broadcasts (logged in)', {
  beforeEach() {
    this.channel = channelScenario(this.server, USERNAME);

    this.video = this.server.create('video', {
      broadcast_type: ARCHIVE_TYPE_PARAM,
      title: TITLE,
      description: DESCRIPTION,
      descriptionHTML: DESCRIPTION,
      descriptionMarkdown: DESCRIPTION,
      tag_list: 'fun,and,games',
      length: DURATION,
      created_at: '2017-01-27T12:14:53Z',
      published_at: null,
      thumbnails: {
        small: [],
        medium: [
          {type: 'generated', url: GENERATED_THUMBNAIL_SRC},
          {type: 'custom', url: CUSTOM_THUMBNAIL_SRC}
        ],
        large: [],
        template: []
      },
      preview: {
        small: '/fixtures/images/stream/1/52x72.jpg',
        medium: '/fixtures/images/stream/1/320x180.jpg'
      },
      views: NUM_VIEWS,
      channel: this.channel
    });

    stubLogin({ login: USERNAME });
    this.page = PastBroadcastsPage.create({ username: USERNAME });
  }
});

test('visiting /:username/manager/past_broadcasts', function(assert) {
  assert.expect(4);
  visit(this.page.url());

  andThen(() => {
    assert.equal(currentURL(), this.page.url());
    let videoElement = find('.card__title:contains(\'My Broadcast\')');
    assert.equal(videoElement.length, 1, "page shows the archived video");
    assert.equal(this.page.headTitle(), 'Video Manager - Twitch', 'has title tag in head');
    assert.notOk(this.page.isFound(this.page.noVideosMessage), 'Does not display no-video message');
  });
});

test('visiting /:username/manager/past_broadcasts with no videos shows no-videos message', function(assert) {
  assert.expect(2);

  // destroy videos
  this.server.schema.videos.all().models.forEach(video => video.destroy());
  assert.equal(this.server.db.videos.length, 0, 'precond - no videos');

  visit(this.page.url());
  andThen(() => {
    assert.ok(this.page.isFound(this.page.noVideosMessage), 'Displays no-video message');
  });
});

test('visiting /:username/manager/past_broadcasts and edit video', function(assert) {
  assert.expect(14);

  visit(this.page.url());
  andThen(() => {
    assert.equal(this.page.videoTitle(), TITLE, 'precond - correct video title');
    assert.equal(this.page.videoDescription(), DESCRIPTION, 'precond - correct video desc');
    assert.ok(this.page.videoInfo().indexOf('Jan 27, 2017') >= 0, 'date is correctly set');
    assert.ok(this.page.videoInfo().indexOf('2:08') >= 0, 'duration is correctly set');
  });
  this.page.clickEdit();
  andThen(() => {
    assert.equal(this.page.editTitleVal(), TITLE, 'edit modal has correct title');
    assert.equal(this.page.editDescriptionVal(), DESCRIPTION, 'edit modal has correct description');
    assert.equal(this.page.editTagsVal(), TAGS, 'edit modal has correct tags');
    assert.equal(this.page.mainThumbnailSrc(), CUSTOM_THUMBNAIL_SRC, 'custom thumbnail is selected');
    assert.equal(this.page.thumbnails().length, 2, 'edit modal has correct number of thumbnails');
  });

  this.page.editVideo({
    title: 'new title',
    descriptionMarkdown: 'new desc',
    tags: 'tag1,tag2,tag3'
  });
  andThen(() => {
    assert.ok(this.page.isFound(this.page.editVideoModalSelector), 'shows edit video modal');
  });
  click(this.page.editVideoDoneButtonSelector);
  andThen(() => {
    let video = this.server.schema.videos.first();
    assert.equal(video.title, 'new title');
    assert.equal(video.description, 'new desc');
    assert.equal(video.tag_list, 'tag1,tag2,tag3');

    assert.equal(this.page.videoTitle(), 'new title');
  });
});

test('visiting /:username/manager/past_broadcasts and delete video', function(assert) {
  assert.expect(2);

  visit(this.page.url());
  click(this.page.moreLinkSelector);
  click(this.page.deleteLinkSelector);
  andThen(() => {
    assert.ok(this.page.isFound(this.page.deleteModalSelector));
  });
  click(this.page.deleteModalDeleteLinkSelector);
  andThen(() => {
    //assert.didNotify('Video successfully deleted');
    assert.equal(this.server.db.videos.length, 0, 'no videos in the db');
    // FIXME should also assert that the video is not displayed in the list of
    // videos on the page -- that functionality appears to be broken at the moment
  });
});

test('visiting /:username/manager/past_broadcasts and create highlight of video', function(assert) {
  assert.expect(2);

  visit(this.page.url());
  click(this.page.highlightSelector);
  andThen(() => {
    assert.equal(currentURL(), this.page.highlightUrl(this.video));
    assert.equal(currentRouteName(), 'manager.highlighter');
  });
});

test('visiting /:username/manager/past_broadcasts when a video is muted', function(assert) {
  assert.expect(1);

  this.mutedVideo = this.server.create('video', {
    title: 'My Muted Stream',
    description: 'totally muted',
    tag_list: 'fun,and,games,muted',
    is_muted: true,
    channel: this.channel
  });
  videoAppealScenario(this.server, this.mutedVideo.id);

  visit(this.page.url());
  click(`.video.item[data-content=${this.mutedVideo.id}] ${this.page.moreLinkSelector}`);
  click(`[data-test-appeal-button]`);

  andThen(() => {
    let appealModal = find('[data-test-appeal-modal]');
    assert.equal(appealModal.length, 1, 'should open the appeal modal');
  });
});

test('visiting /:username/manager/past_broadcasts has expected options on video card', function(assert) {
  assert.expect(8);

  visit(this.page.url());
  andThen(() => {
    let editButton = find('[data-test-selector=edit-button]');
    let moreMenu = find('[data-test-selector=more-menu]');
    let moreButton = moreMenu.find('[data-test-selector=more-menu-button]');
    let highlightButton = moreMenu.find('a:contains(Highlight)');
    let downloadButton = moreMenu.find('a:contains(Download)');
    let deleteButton = moreMenu.find('[data-test-selector=delete-button]');
    let exportButton = moreMenu.find('[data-test-selector=export-button]');
    let videoStatsButton = moreMenu.find('[data-test-selector=video-stats-button]');

    assert.equal(editButton.length, 1, 'Edit button present');
    assert.equal(moreMenu.length, 1, 'More menu present');
    assert.equal(moreButton.length, 1, 'More button present');
    assert.equal(highlightButton.length, 1, 'More menu should have highlight button');
    assert.equal(downloadButton.length, 1, 'More menu should have download button');
    assert.equal(deleteButton.length, 1, 'More menu should have delete button');
    assert.equal(exportButton.length, 1, 'More menu should have export button');
    assert.equal(videoStatsButton.length, 1, 'More menu should have video stats button');
  });
});
