import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';
import videoScenario from 'web-client/mirage/scenarios/video';
import VideoPage from 'web-client/tests/pages/videos';
import { UPLOAD_TYPE_PARAM } from 'web-client/utilities/video/types';
import channelScenario from 'web-client/mirage/scenarios/channel';
import sinon from 'sinon';

moduleForAcceptance('Acceptance | videos', {
  beforeEach() {
    this.video = videoScenario(this.server, { channel: channelScenario(this.server) });
    this.page = VideoPage.create({ vodId: this.video.id });
    this.rechatService = this.owner.lookup('service:vod-chat-service');
  }
});

test('visiting a VOD watching page', function(assert) {
  assert.expect(6);

  visit(this.page.url());

  andThen(() => {
    assert.equal(currentURL(), `/videos/${this.video.id}`);
    assert.equal(currentRouteName(), 'videos');
    assert.equal(this.page.canonicalUrl(), `https://www.twitch.tv/videos/${this.video.id}`, 'Canonical URL present');
    assert.equal(this.page.headTitle(), `twitch - Twitch`, 'has title tag in head');

    let rightColumn = find(this.page.rightColumn);
    assert.notOk(rightColumn.hasClass('closed'), 'the right column should not be hidden');

    let rechatContainer = find(this.page.rechatContainer);
    assert.equal(rechatContainer.length, 1, '.js-rightcol-content .js-chat-container renders');
  });
});

test('visiting a VOD page for an uploaded video', function(assert) {
  assert.expect(4);

  let uploadedVideo = videoScenario(this.server, {
    broadcast_type: UPLOAD_TYPE_PARAM
  });

  let uploadedVideoPage = VideoPage.create({ vodId: uploadedVideo.id });
  visit(uploadedVideoPage.url());

  andThen(() => {
    assert.equal(currentURL(), `/videos/${uploadedVideo.id}`);
    assert.equal(currentRouteName(), 'videos');

    let rightColumn = find(this.page.rightColumn);
    assert.notOk(rightColumn.hasClass('closed'), 'the right column should not be hidden');

    let followColumn = find(uploadedVideoPage.followColumn);
    assert.equal(followColumn.length, 1, 'should show the follow column and not rechat');
  });
});

test('visiting a VOD page for a non-existent video', function(assert) {
  assert.expect(6);

  let vodId = '404';
  let page = VideoPage.create({ vodId: vodId });

  visit(page.url());

  andThen(() => {
    assert.equal(currentURL(), `/videos/${vodId}`);
    assert.equal(currentRouteName(), 'videos');
    assert.equal(this.page.headTitle(), `Twitch`, 'has title tag in head');

    let rightColumn = find(this.page.rightColumn);
    assert.notOk(rightColumn.hasClass('closed'), 'the right column should not be hidden');

    let followColumn = find(this.page.followColumn);
    assert.equal(followColumn.length, 1, 'should show the follow column and not rechat');

    assert.ok(page.noChannelMessage(), 'should render the no channel message');
  });
});

test('visiting a VOD page for an existing VOD, but missing channel', function(assert) {
  assert.expect(6);

  let channel = channelScenario(this.server, '404');
  let uploadedVideo = videoScenario(this.server, {
    broadcast_type: UPLOAD_TYPE_PARAM,
    channel
  });

  let uploadedVideoPage = VideoPage.create({ vodId: uploadedVideo.id });
  visit(uploadedVideoPage.url());

  andThen(() => {
    assert.equal(currentURL(), `/videos/${uploadedVideo.id}`);
    assert.equal(currentRouteName(), 'videos');
    assert.equal(this.page.headTitle(), `Twitch`, 'has title tag in head');

    let rightColumn = find(this.page.rightColumn);
    assert.notOk(rightColumn.hasClass('closed'), 'the right column should not be hidden');

    let followColumn = find(this.page.followColumn);
    assert.equal(followColumn.length, 1, 'should show the follow column and not rechat');

    assert.ok(this.page.noChannelMessage(), 'should render the no channel message');
  });
});

test('visiting a VOD page with a collection queryparam set', function(assert) {
  assert.expect(3);

  let mainRouter = this.owner.lookup('router:main');
  let sendSpy = sinon.spy(mainRouter, 'send');

  let collectionId = "asdf1234567890";
  let videoId = '12125189124';

  let url = `/videos/${videoId}?collection=${collectionId}`;

  sendSpy.withArgs('setCollection', collectionId);

  visit(url);

  andThen(() => {
    assert.equal(currentURL(), `/videos/${videoId}?collection=${collectionId}`);
    assert.equal(currentRouteName(), 'videos');

    assert.ok(sendSpy.calledWith('setCollection', collectionId), 'sends setCollection if queryParam is present');
  });
});

test('visiting the VOD page with the Clips on Channel Page experiment enabled', function(assert) {
  assert.expect(1);

  this.experiments.use({ CLIPS_FEEL_IT_COMING: 'variant1' });

  visit(this.page.url());

  andThen(() => {
    assert.elementCount('[data-test-selector="channel-clips-tab"]', 1, 'Clips Tab is rendered');
  });
});
