import QUnit from 'qunit';
import $ from 'jquery';

/*
  Assert that a css selector text contains a string.

  Example:

  ```js
  assert.elementText('.selector', 'the string', 'this is the message');
  ```

  -OR-

  ```js
  assert.elementText('.selector', /12\s+Videos/, 'has 12 videos');
  ```

 @method assert.elementText
 @param {String} selector The css selector to get a DOM element
 @param {String | Object} expectedText The string that should be found in selector, or a regular expression Object
*/
QUnit.assert.elementText = function(selector, expectedText, message) {
  let actualText = $(selector).text().trim().replace(/\s+/g, ' ');
  message = message || `${actualText} did not match ${expectedText} with assert.elementText`;

  if (expectedText instanceof RegExp) {
    this.ok(actualText.match(expectedText), message);
  } else if (typeof expectedText === 'string') {
    this.equal(actualText, expectedText, message);
  } else {
    this.ok(false, 'assert.elementText expects a string or RegExp');
  }
};

/*
    Assert that a css selector val contains a string.

  Example:

  ```js
  assert.elementVal('.selector', 'the string', 'and a message');
  ```

  @method assert.elementVal
  @param {String} selector The css selector to get a DOM element
  @param {String} expectedVal The string that should be found in selector
*/
QUnit.assert.elementVal = function(selector, expectedVal, message) {
  let actualVal = $(selector).val().trim();
  this.equal(actualVal, expectedVal, message);
};

/*
    Assert that a css selector has a number of elements in the DOM

  Example:

  ```js
  assert.elementCount('.selector', 3, 'with a message');
  ```

  @method assert.elementCount
  @param {String} selector The css selector to get a DOM element
  @param {String} expectedCount The expected element count
*/
QUnit.assert.elementCount = function(selector, expectedCount, message) {
  let elementCount = $(selector).length;
  this.equal(elementCount, expectedCount, message);
};


/*
    Assert that an element has the expected width

  Example:

  ```js
  assert.elementWidth('.selector', 3, 'width is correct');
  ```

  @method assert.elementWidth
  @param {String} selector The css selector to get a DOM element
  @param {String} expectedWidth The expected element width
*/
QUnit.assert.elementWidth = function(selector, expectedWidth, message) {
  let element = $(selector);
  if (element.length === 0) {
    this.ok(false, `could not find element width selector ${selector}`);
  }

  let elementWidth = element[0].getBoundingClientRect().width;
  this.equal(elementWidth, expectedWidth, message);
};

QUnit.assert.elementHasClass = function(selector, expectedClass, message) {
  let element = $(selector);
  if (element.length === 0) {
    this.ok(false, `could not find element for selector ${selector}`);
    return;
  }

  this.ok(element.hasClass(expectedClass), message);
};

QUnit.assert.notElementHasClass = function(selector, expectedClass, message) {
  let element = $(selector);
  if (element.length === 0) {
    this.ok(false, `could not find element for selector ${selector}`);
    return;
  }

  this.notOk(element.hasClass(expectedClass), message);
};
