/* globals Twitch */

import { test } from 'ember-qunit';
import moduleForComponent from 'web-client/tests/helpers/module-for-component';
import hbs from 'htmlbars-inline-precompile';

moduleForComponent('backpack/payment-info-button', 'Integration | Component | backpack/payment info button', {
  integration: true
});

test('Extend backpack/support-button', function(assert) {
  assert.expect(3);

  this.set('openedBalloon', 'paymentInfo');
  this.set('toggleBalloon', (name) => {
    assert.equal(`Toggling ${name}`, 'Toggling paymentInfo', 'toggleBalloon is invoked');
  });

  this.render(hbs`
    {{backpack/payment-info-button openedBalloon=openedBalloon toggleBalloon=toggleBalloon}}
  `);

  assert.ok($('.balloon-wrapper.profile-actions__button'), 'classNames is correct');
  assert.ok($('.balloon.show'), 'openedBalloon is true');

  // this does another assert
  this.$().find('button').click();
});

test('Content verification', function(assert) {
  assert.expect(3);

  this.set('ticket', {
    'product': {
      'recurring': true,
      'period': 'Month',
      'interval_number': 1,
      'price': 4.99
    },
    'purchase_profile': {
      'paid_on': '2016-01-01T12:00:00Z'
    }
  });

  this.render(hbs`
    {{backpack/payment-info-button ticket=ticket}}
  `);

  assert.equal(this.$('button').text().trim(), 'Payment Info', 'Button label');
  assert.equal(this.$('.balloon .mg-b-1:nth-child(1)').text().trim(), 'Recurring Payment Date: 01/02', 'Price label');
  assert.equal(this.$('.balloon .mg-b-1:nth-child(2)').text().trim(), '4.99 / Month', 'Cadence');
});

test('Balloon links - expired', function(assert) {
  assert.expect(3);

  this.set('ticket', {
    'product': {
      'short_name': 'awesomechannel'
    },
    'purchase_profile': {
      'paid_on': '2016-01-01T00:00:00Z',
      'will_renew': false,
      'expired': true,
      'refundable': false
    }
  });

  this.render(hbs`
    {{backpack/payment-info-button ticket=ticket}}
  `);

  assert.equal(this.$('.balloon a').length, 1, 'Only cancel link');
  assert.equal(this.$('.balloon a').attr('href'), Twitch.uri.productCancel('awesomechannel'), 'Cancel link href');
  assert.equal(this.$('.balloon a').text().trim(), 'Cancel', 'Cancel link text');
});

test('Balloon links - renewable and not refundable', function(assert) {
  assert.expect(3);

  this.set('ticket', {
    'product': {
      'short_name': 'awesomechannel'
    },
    'purchase_profile': {
      'paid_on': '2016-01-01T00:00:00Z',
      'will_renew': true,
      'expired': false,
      'refundable': false
    }
  });

  this.render(hbs`
    {{backpack/payment-info-button ticket=ticket}}
  `);

  assert.equal(this.$('.balloon a').length, 1, 'Only do not renew link');
  assert.equal(this.$('.balloon a').attr('href'), Twitch.uri.productDoNotRenew('awesomechannel'), 'Do not renew link href');
  assert.equal(this.$('.balloon a').text().trim(), "Don't Renew", 'Do not renew link text');
});

test('Balloon links - renewable and refundable', function(assert) {
  assert.expect(4);

  this.set('ticket', {
    'product': {
      'short_name': 'awesomechannel'
    },
    'purchase_profile': {
      'paid_on': '2016-01-01T00:00:00Z',
      'will_renew': true,
      'expired': false,
      'refundable': true
    }
  });

  this.render(hbs`
    {{backpack/payment-info-button ticket=ticket}}
  `);

  assert.equal(this.$('.balloon a').length, 2, 'Two buttons show up');
  assert.equal(this.$('.balloon a:nth-of-type(2)').attr('href'), Twitch.uri.productCancelAndRefund('awesomechannel'), 'Cancel and refund link href');
  assert.equal(this.$('.balloon a:nth-of-type(2)').text().trim(), 'Cancel', 'Cancel and refund link text');
  assert.equal(this.$('.balloon').children().remove().end().text().trim(), 'or', 'Has or delimiter');
});

test('Balloon links - refundable and not renewable', function(assert) {
  assert.expect(3);

  this.set('ticket', {
    'product': {
      'short_name': 'awesomechannel'
    },
    'purchase_profile': {
      'paid_on': '2016-01-01T00:00:00Z',
      'will_renew': false,
      'expired': false,
      'refundable': true
    }
  });

  this.render(hbs`
    {{backpack/payment-info-button ticket=ticket}}
  `);

  assert.equal(this.$('.balloon a').length, 1, 'Only refundable link');
  assert.equal(this.$('.balloon a').attr('href'), Twitch.uri.productCancelAndRefund('awesomechannel'), 'Refundable link href');
  assert.equal(this.$('.balloon a').text().trim(), 'Cancel', 'Refundable link text');
});
