/* globals Twitch */
import moduleForComponent from 'web-client/tests/helpers/module-for-component';
import { test } from 'ember-qunit';
import hbs from 'htmlbars-inline-precompile';
import wait from 'ember-test-helpers/wait';
import RSVP from 'rsvp';

const originalTwitchGeo = Twitch.geo;

moduleForComponent('broadcaster-language-picker', 'Integration | Component | broadcaster-language-picker', {
  beforeEach() {
    // prevent using test harnesses default browser
    Twitch.geo = RSVP.resolve({
      preferred_language: ''
    });
  },

  afterEach() {
    Twitch.geo = originalTwitchGeo;
  }
});

test('renders available broadcaster languages', function (assert) {
  this.inject.service('globals');
  let languages = this.globals.get('broadcasterLanguages');

  this.render(hbs`{{broadcaster-language-picker}}`);

  assert.expect(Object.keys(languages).length);

  for (let languageCode in languages) {
    let languageText = languages[languageCode];
    let option = this.$(`option[value=${languageCode}]`);

    assert.ok(
      option.text().indexOf(languageText) > -1,
      `${languageCode} is found and contains correct text (${languageText})`
    );
  }
});

test('includes a prompt if no languageCode or browser default is provided', function (assert) {
  assert.expect(1);
  let done = assert.async();
  this.render(hbs`{{broadcaster-language-picker languageCode=null}}`);

  Twitch.geo.then(() => {
    let select = this.$('select')[0];
    let options = this.$('option');
    let selected = options[select.selectedIndex];

    assert.equal(selected.text, 'Pick a Language', 'prompt is properly selected');

    done();
  });
});

test('selects the specified languageCode if provided', function (assert) {
  assert.expect(2);
  let done = assert.async();

  this.render(hbs`{{broadcaster-language-picker languageCode="en"}}`);

  Twitch.geo.then(() => {
    let select = this.$('select')[0];
    let options = this.$('option');
    let selected = options[select.selectedIndex];

    assert.equal(selected.text, 'English', 'prompt is properly selected');

    let inputCode = this.$('input[name="broadcaster_language"]').val();
    assert.equal(inputCode, 'en', 'language code is properly set');

    done();
  });
});

test('selects the browser default language if languageCode is not present', function (assert) {
  assert.expect(2);
  let done = assert.async();
  let code = 'da';

  Twitch.geo = RSVP.resolve({
    preferred_language: code
  });

  this.render(hbs`{{broadcaster-language-picker}}`);

  Twitch.geo.then(() => {
    let select = this.$('select')[0];
    let options = this.$('option');
    let selected = options[select.selectedIndex];

    assert.equal(selected.text, 'Dansk', 'prompt is properly selected');

    let inputCode = this.$('input[name="broadcaster_language"]').val();
    assert.equal(inputCode, code, 'language code is properly set');
    done();
  });
});

test('selects the specified languageCode over the browser default language', function (assert) {
  assert.expect(2);
  let done = assert.async();

  Twitch.geo = RSVP.resolve({
    preferred_language: 'da'
  });

  this.render(hbs`{{broadcaster-language-picker languageCode='no'}}`);

  Twitch.geo.then(() => {
    let select = this.$('select')[0];
    let options = this.$('option');
    let selected = options[select.selectedIndex];

    assert.equal(selected.text, 'Norsk', 'prompt is properly selected');

    let inputCode = this.$('input[name="broadcaster_language"]').val();
    assert.equal(inputCode, 'no', 'language code is properly set');
    done();
  });
});

test('sets the value on a hidden input when the user selects a language (legacy behavior)', function (assert) {
  assert.expect(1);

  this.render(hbs`{{broadcaster-language-picker}}`);
  let select = this.$('select');
  let input = this.$('input[name="broadcaster_language"]');

  select.val('asl');
  select.change();

  return wait().then(() => {
    assert.equal(input.val(), 'asl', 'language code was updated');
  });
});

test('calls the onChange action if provided', function (assert) {
  assert.expect(1);

  this.render(hbs`{{broadcaster-language-picker onChange=(action (mut onChangeValue))}}`);
  let select = this.$('select');

  select.val('asl');
  select.change();

  assert.equal(this.get('onChangeValue'), 'asl', 'language code was updated');
});
