import { test } from 'ember-qunit';
import moduleForComponent from 'web-client/tests/helpers/module-for-component';
import hbs from 'htmlbars-inline-precompile';
import { clipModel } from 'web-client/mirage/fixtures/clips';
import sinon from 'sinon';
import * as utils from 'ember-metal/utils';
import { commatized } from 'web-client/helpers/commatized';
import Service from 'ember-service';
import { assign } from 'ember-platform';

moduleForComponent('clips/clips-modal-playback', 'Integration | Component | clips-modal-playback', {
  integration: true,
  beforeEach() {
    this.set('clip', clipModel);
    const clips = {};
    clips.items = [clipModel, clipModel];
    this.set('clips', clips);
    this.pauseSpy = sinon.spy();
    this.defaultTemplate = hbs`{{clips/clips-modal-playback clip clips=clips}}`;
    this.register('service:persistent-player', Service.extend({
      playerComponent: {
        pause: this.pauseSpy
      }
    }));

    this.clipInfoSelector = '[data-test-selector="clip-modal-card-info"]';
  }
});

test('when no "clip" is provided', function (assert) {
  assert.expect(1);
  sinon.stub(utils, 'assert');
  this.render(hbs`{{clips/clips-modal-playback}}`);

  assert.equal(utils.assert.getCall(0).args[1], false, 'assert called with false');

  utils.assert.restore();
});

test('clip embed iframe', function (assert) {
  assert.expect(2);
  this.render(this.defaultTemplate);

  let $embedIframe = this.$('[data-test-selector="clips-playback"]');

  assert.equal($embedIframe.attr('src'), clipModel.embedURL, 'embed player rendered');
  assert.equal($embedIframe.attr('allowfullscreen'), 'true', 'embed player can be made fullscreen');
});

test('clip card info', function (assert) {
  assert.expect(6);
  this.render(this.defaultTemplate);

  assert.elementText(
    this.clipInfoSelector,
    new RegExp(`${clipModel.broadcasterDisplayName} streaming ${clipModel.game}`),
    'renders broadcaster and game information'
  );
  assert.elementText(this.clipInfoSelector, new RegExp(`Clipped by ${clipModel.curatorDisplayName}`), 'renders curator info');
  assert.elementText(
    '[data-test-selector="clip-modal-viewcount"]',
    new RegExp(`${commatized(clipModel.views)} plays`),
    'renders viewcount'
  );

  assert.elementCount('[data-test-selector="clips-share-box"]', 1, 'renders clips share box');
  assert.elementCount(`#share-url-copy`, 1, 'renders share URL textbox');
  assert.equal(this.$('#share-url-copy').attr('value'), clipModel.url, 'share URL textbox contains clip URL');
});

test('on modal playback initialization ', function (assert) {
  assert.expect(1);
  this.render(this.defaultTemplate);

  assert.ok(this.pauseSpy.calledOnce, 'calls pause method on persistentPlayer service');
});

test('clip card info when game is not present', function (assert) {
  assert.expect(1);
  this.render(this.defaultTemplate);

  this.set('clip', assign({}, clipModel, { game: '' }));

  assert.elementText(
    this.clipInfoSelector,
    new RegExp(clipModel.broadcasterDisplayName),
    'renders broadcaster but no game information'
  );
});

test('more button', function (assert) {
  assert.expect(1);
  this.render(this.defaultTemplate);

  assert.elementCount('[data-test-selector="clips-more-button-root"]', 1, 'renders the root component of more button');
});

test('arrows with multiple clips', function (assert) {
  assert.expect(2);
  this.render(this.defaultTemplate);

  assert.elementCount('[data-test-selector="clip-modal-arrow-left"]', 0, 'does not render the left arrow on first clip');
  assert.elementCount('[data-test-selector="clip-modal-arrow-right"]', 1, 'always renders the right arrow');
});

test('arrows with one clip', function (assert) {
  assert.expect(2);
  const clips = {};
  clips.items = [clipModel];
  this.set('clips', clips);
  this.render(this.defaultTemplate);

  assert.elementCount('[data-test-selector="clip-modal-arrow-left"]', 0, 'does not render the left arrow with one clip');
  assert.elementCount('[data-test-selector="clip-modal-arrow-right"]', 0, 'does not render the left arrow with one clip');
});
