import { test } from 'ember-qunit';
import sinonTest from 'ember-sinon-qunit/test-support/test';
import moduleForComponent from 'web-client/tests/helpers/module-for-component';
import RSVP from 'rsvp';
import EmberObject from 'ember-object';
import wait from 'ember-test-helpers/wait';
import hbs from 'htmlbars-inline-precompile';
import { loadGameDetail, loadOffer } from 'web-client/tests/integration/components/directory/game-details/game-detail-helper';

moduleForComponent('commerce/channel-box', 'Integration | Component | commerce/channel-box', {
  integration: true,
  beforeEach() {
    this.store = this.owner.lookup('service:store');
    this.gameSettings = this.owner.lookup('service:gameSettings');
    this.loadGameDetail = loadGameDetail;
    this.loadOffer = loadOffer;
  }
});

test('it renders as hidden until data is loaded', function(assert) {
  this.render(hbs`{{commerce/channel-box}}`);

  assert.equal(this.$('.qa-cmrc-channel-box').length, 0, 'Immediately after rendering, component is hidden');

});


sinonTest('it loads details after loading', function() {
  this.set('channel', {
    game: 'Integration Test'
  });
  let mockDetails = new EmberObject({
    product: {
      title: 'Integration Test'
    }
  });
  let mock = this.mock(this.store);
  mock.expects('queryRecord')
    .once()
    .withArgs('game-detail', {game:'Integration Test'})
    .returns(RSVP.resolve(mockDetails));

  this.render(hbs`{{commerce/channel-box channel=channel}}`);

  return wait().then(() => {
    mock.verify();
  });

});

sinonTest('it doesnt load details if gameDetails is same as channel.game', function(){
  this.set('channel', {
    game: 'Integration Test'
  });
  this.set('mockDetails', new EmberObject({
    product: {
      title: 'Integration Test'
    }
  }));

  let mock = this.mock(this.store);
  mock.expects('queryRecord')
    .never();

  this.render(hbs`{{commerce/channel-box channel=channel gameDetails=mockDetails}}`);

  return wait().then(() => {
    mock.verify();
  });
});

/**
 * Please note: This functionality should be temporary, once we can handle load, we don't want to gate
 * calls on getGameSettings
 */
sinonTest('it doesnt load details if gameSettings is false', function(){
  this.set('channel', {
    game: 'Integration Test'
  });

  let mock = this.mock(this.store);
  mock.expects('queryRecord')
    .never();
  let mockGameSettings = this.mock(this.gameSettings);
  mockGameSettings.expects('getGameSettings')
    .once()
    .returns(RSVP.resolve(false));

  this.render(hbs`{{commerce/channel-box channel=channel gameDetails=mockDetails}}`);

  return wait().then(() => {
    mockGameSettings.verify();
    mock.verify();
  });
});

sinonTest('it doesnt load details if channel is hosting', function() {
  this.set('channel', {
    game: 'Integration Test'
  });
  this.set('isHosting', 'AnAwesomeChannel');


  let mock = this.mock(this.store);
  mock.expects('queryRecord')
    .never();

  this.render(hbs`{{commerce/channel-box channel=channel isHosting=isHosting}}`);

  return wait().then(() => {
    mock.verify();
  });
});

sinonTest('it hides if there arent gameDetails', function(assert) {
  this.set('channel', {
    game: 'Integration Test'
  });
  let mock = this.mock(this.store);
  mock.expects('queryRecord')
    .once()
    .withArgs('game-detail', {game:'Integration Test'})
    .returns(RSVP.reject());

  this.render(hbs`{{commerce/channel-box channel=channel}}`);

  return wait().then(() => {
    mock.verify();
    assert.equal(this.$('.qa-cmrc-channel-box').length, 0, 'channel box is still hidden');
  });
});

sinonTest('it hides if the game isnt for sale', function(assert) {
  this.set('channel', {
    game: 'Integration Test',
    partner: true
  });
  let mockDetails = new EmberObject({
    product: {
      title: 'Integration Test'
    },
    isForSale: false
  });
  let mock = this.mock(this.store);
  mock.expects('queryRecord')
    .once()
    .withArgs('game-detail', {game:'Integration Test'})
    .returns(RSVP.resolve(mockDetails));

  this.render(hbs`{{commerce/channel-box channel=channel}}`);

  return wait().then(() => {
    mock.verify();
    assert.equal(this.$('.qa-cmrc-channel-box').length, 0, 'channel box is still hidden');
  });
});

sinonTest('it hides if the game is for sale, but not a partner', function(assert) {
  this.set('channel', {
    game: 'Integration Test',
    partner: false
  });
  let mockDetails = new EmberObject({
    product: {
      title: 'Integration Test'
    },
    isForSale: true
  });
  let mock = this.mock(this.store);
  mock.expects('queryRecord')
    .once()
    .withArgs('game-detail', {game:'Integration Test'})
    .returns(RSVP.resolve(mockDetails));

  this.render(hbs`{{commerce/channel-box channel=channel}}`);

  return wait().then(() => {
    mock.verify();
    assert.equal(this.$('.qa-cmrc-channel-box').length, 0, 'channel box is still hidden');
  });
});

sinonTest('it unhides self if game is for sale and a partner', function(assert) {
  this.set('channel', {
    game: 'Integration Test',
    partner: true
  });
  let mockDetails = new EmberObject({
    product: {
      title: 'Integration Test'
    },
    isForSale: true
  });
  let mock = this.mock(this.store);
  mock.expects('queryRecord')
    .once()
    .withArgs('game-detail', {game:'Integration Test'})
    .returns(RSVP.resolve(mockDetails));

  this.render(hbs`{{commerce/channel-box channel=channel}}`);

  return wait().then(() => {
    mock.verify();
    assert.equal(this.$('.qa-cmrc-channel-box').length, 1, 'channel box is not hidden');

  });

});
