import moduleForComponent from 'web-client/tests/helpers/module-for-component';
import { test } from 'ember-qunit';
import hbs from 'htmlbars-inline-precompile';
import sinon from 'sinon';

const CHECKBOX = '.simple-checkbox__input';

const sandbox = sinon.sandbox.create();
const toggleProperty = function(keyName) {
  this.set(keyName, !this.get(keyName));
};

moduleForComponent('common/simple-checkbox', 'Integration | Component | common/simple-checkbox', {
  afterEach() {
    sandbox.restore();
  }
});

test('renders a checkbox (noop handler)', function(assert) {
  const update = sinon.stub();
  const model = { checked: true };

  this.set('update', update);
  this.render(hbs`
    {{common/simple-checkbox
        update=(action update 'model.checked')
        checked=(readonly model.checked)
    }}
  `);

  this.$(CHECKBOX).click();
  assert.ok(update.calledOnce, `should call 'update' exactly once`);
  assert.strictEqual(model.checked, true, 'should not modify the value');
});

test('renders a checkbox (with click handlers)', function(assert) {
  const update = toggleProperty.bind(this);
  const model = { checked: false };

  this.setProperties({ model,  update });
  this.render(hbs`
    {{common/simple-checkbox
        update=(action update 'model.checked')
        checked=(readonly model.checked)
    }}
  `);

  assert.strictEqual(this.$(CHECKBOX).prop('checked'), false, `should set render the correct initial value`);

  this.$(CHECKBOX).click();
  assert.strictEqual(model.checked, true, `should properly set the value to 'true'`);
  assert.ok(this.$(CHECKBOX).prop('checked'), `should the checkbox as 'checked'`);

  this.$(CHECKBOX).click();
  assert.strictEqual(model.checked, false, `should properly handle the value to 'false'`);
  assert.notOk(this.$(CHECKBOX).prop('checked'), `should the checkbox as 'unchecked'`);
});

test('renders a checkbox (disabled)', function(assert) {
  const update = sinon.spy(toggleProperty.bind(this));
  const model = { checked: true };

  this.setProperties({
    model,
    update,
    disabled: true
  });
  this.render(hbs`
    {{common/simple-checkbox
        update=(action update 'model.checked')
        checked=(readonly model.checked)
        disabled=disabled
    }}
  `);

  this.$(CHECKBOX).click();
  assert.ok(update.notCalled, `should never call 'update' when disabled=true`);
  assert.strictEqual(model.checked, true, 'should not modify the value');

  this.set('disabled', false);
  this.$(CHECKBOX).click();
  assert.ok(update.calledOnce, `should call 'update' exactly once when disabled=false`);
  assert.strictEqual(model.checked, false, `should update the model with the correct value`);
});

test('renders a checkbox (readonly)', function(assert) {
  const update = sinon.spy(toggleProperty.bind(this));
  const model = { checked: true };

  this.setProperties({
    model,
    update,
    readonly: true
  });
  this.render(hbs`
    {{common/simple-checkbox
        update=(action update 'model.checked')
        checked=(readonly model.checked)
        readonly=readonly
    }}
  `);

  this.$(CHECKBOX).click();
  assert.ok(update.notCalled, `should never call 'update' when readonly=true`);
  assert.strictEqual(model.checked, true, 'should not modify the value');

  this.set('readonly', false);
  this.$(CHECKBOX).click();
  assert.ok(update.calledOnce, `should call 'update' exactly once when readonly=false`);
  assert.strictEqual(model.checked, false, `should update the model with the correct value`);
});

test('renders a checkbox (custom content)', function(assert) {
  const update = sinon.spy(toggleProperty.bind(this));
  const contentClassName = 'some_custom_class';
  const model = { checked: true };

  this.setProperties({ contentClassName, model, update });
  this.render(hbs`
    {{#common/simple-checkbox update=(action update 'model.checked') checked=(readonly model.checked)}}
      <span class={{contentClassName}}>
        <img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQBAMAAADt3eJSAAAABGdBTUEAALGPC/xhBQAAACBjSFJNAAB6JgAAgIQAAPoAAACA6AAAdTAAAOpgAAA6mAAAF3CculE8AAAAGFBMVEX///9kQaVkQaVkQaVkQaVkQaVkQaX////s0MlGAAAABnRSTlMBXWV9JU87OK+aAAAAAWJLR0QAiAUdSAAAAAd0SU1FB+AJDwczKTx1KNwAAAA9SURBVAjXY2BIgwAGRhiDLY0BBFAZTAlqBBnIuphhDLMUICMtAYgdwIy0tBQGECMNJMAAYqaCNQIZAUASABQiF8Yh4KvNAAAAJXRFWHRkYXRlOmNyZWF0ZQAyMDE2LTA5LTE1VDA3OjUxOjQyKzAyOjAwcQ6YuwAAACV0RVh0ZGF0ZTptb2RpZnkAMjAxNi0wOS0xNVQwNzo1MTo0MSswMjowMDG7OpoAAAAASUVORK5CYII=' />
        Twitch: Lorem ipsum dolor sit amet, consectetur adipiscing elit. Aliquam tincidunt aliquet volutpat.
      </span>
    {{/common/simple-checkbox}}
  `);

  this.$(`.${contentClassName}`).click();
  assert.ok(update.calledOnce, `should call 'update' exactly once`);
  assert.strictEqual(model.checked, false, `should update the model with the correct value`);
  assert.elementText(`.${contentClassName}`, 'Twitch: Lorem ipsum dolor sit amet, consectetur adipiscing elit. Aliquam tincidunt aliquet volutpat.');
});
