import moduleForComponent from 'web-client/tests/helpers/module-for-component';
import { test } from 'ember-qunit';
import hbs from 'htmlbars-inline-precompile';
import RSVP from 'rsvp';
import moment from 'moment';

moduleForComponent('dashboards/events/create-event-form', 'Integration | Component | dashboards/events/create-event-form', {
  integration: true,

  beforeEach() {
    this.onDidSaveCalls = 0;
    this.set('onDidSave', () => {
      this.onDidSaveCalls++;
    });

    this.routing = this.owner.lookup('service:routing');
  }
});

test('Calls onDidSave after a successful save', function(assert) {
  assert.expect(1);

  this.set('event', {
    save() {
      return RSVP.resolve();
    }
  });

  this.render(hbs`
    {{#dashboards/events/create-event-form event=event onDidSave=onDidSave}}
      <button type="submit">Submit</button>
    {{/dashboards/events/create-event-form}}
  `);

  this.$('button').click();

  assert.equal(this.onDidSaveCalls, 1, 'calls onDidSave');
});

test('Shows an error message when saving fails', function(assert) {
  assert.expect(4);

  this.set('event', {
    save() {
      return RSVP.reject({ status: 500 });
    }
  });

  this.render(hbs`
    {{#dashboards/events/create-event-form event=event onDidSave=onDidSave as |form|}}
      {{#if form.didReceiveErrorResponse}}
        <div id="some-error"></div>
      {{/if}}

      <button type="submit">Submit</button>
    {{/dashboards/events/create-event-form}}
  `);

  this.$('button').click();

  assert.equal(this.onDidSaveCalls, 0, 'does not call onDidSave');
  assert.equal(this.$('#some-error').length, 1, 'shows an error');

  this.set('event', {
    save() {
      return RSVP.resolve();
    }
  });

  this.$('button').click();

  assert.equal(this.onDidSaveCalls, 1, 'calls onDidSave');
  assert.equal(this.$('#some-error').length, 0, 'hides the error');
});

test('handling onGameSelect with an existing game', function(assert) {
  assert.expect(3);

  this.set('game', { name: 'Game', id: 5 });
  this.set('event', {});

  this.render(hbs`
    {{#dashboards/events/create-event-form event=event as |form|}}
      <button type="button" onclick={{action form.onGameSelect game}}>Click!</button>
      <div id="errors">{{form.validation.errors.game}}</div>
      <div id="game-name">{{form.gameName}}</div>
    {{/dashboards/events/create-event-form}}
  `);

  this.$('button').click();

  assert.equal(this.$('#game-name').text(), 'Game', 'sets the form gameName');
  assert.equal(this.get('event.gameId'), 5, 'sets the gameId');
  assert.equal(this.$('#errors').text(), '', 'shows no errors');
});

test('handling onGameSelect with empty game', function(assert) {
  assert.expect(1);

  this.set('event', {});
  this.set('game', { name: '', id: null });

  this.render(hbs`
    {{#dashboards/events/create-event-form event=event as |form|}}
      <button type="button" onclick={{action form.onGameSelect game}}>Click!</button>
      <div id="errors">{{form.validation.errors.game}}</div>
    {{/dashboards/events/create-event-form}}
  `);

  this.$('button').click();

  assert.equal(this.$('#errors').text(), '', 'shows no errors');
});

test('handling onGameSelect with missing game', function(assert) {
  assert.expect(1);

  this.set('event', {});
  this.set('game', { name: 'Missing', id: null });

  this.render(hbs`
    {{#dashboards/events/create-event-form event=event as |form|}}
      <button type="button" onclick={{action form.onGameSelect game}}>Click!</button>
      <div id="errors">{{form.validation.errors.game}}</div>
    {{/dashboards/events/create-event-form}}
  `);

  this.$('button').click();

  assert.equal(this.$('#errors').text(), 'Please choose a supported game', 'shows errors');
});

test('validating an image onImagePicked', function(assert) {
  assert.expect(1);

  this.set('event', {});
  this.set('getDataURL', () => RSVP.Promise.resolve('some-url'));
  this.set('image', 'some-file');

  this.render(hbs`
    {{#dashboards/events/create-event-form event=event getDataURL=getDataURL as |form|}}
      <button type="button" onclick={{action form.onImagePicked image}}>Click!</button>
      <div id="errors">{{form.validation.warnings.imageFile}}</div>
    {{/dashboards/events/create-event-form}}
  `);

  this.$('button').click();

  assert.stringIncludes(this.$('#errors').text(), 'must be', 'shows errors');
});

test('cannot submit while game is pending', function(assert) {
  assert.expect(2);

  this.set('event', {
    startTime: moment().add({ day: 1 }),
    endTime: moment().add({ day: 1, hour: 1 })
  });

  this.render(hbs`
    {{#dashboards/events/create-event-form event=event as |form|}}
      <button id="onGamePending" type="button" onclick={{action form.onGamePending}}>Pending</button>
      <button id="onGameSelect" type="button" onclick={{action form.onGameSelect}}>Select</button>
      <div id="isDisabled">{{form.isDisabled}}</div>
    {{/dashboards/events/create-event-form}}
  `);

  this.$('#onGamePending').click();

  assert.equal(this.$('#isDisabled').text(), 'true', 'isDisabled is true');

  this.$('#onGameSelect').click();

  assert.equal(this.$('#isDisabled').text(), 'false', 'isDisabled is false');
});
