import Object from 'ember-object';
import moduleForComponent from 'web-client/tests/helpers/module-for-component';
import { test } from 'ember-qunit';
import hbs from 'htmlbars-inline-precompile';
import wait from 'ember-test-helpers/wait';
import MockAutohostService from 'web-client/tests/helpers/mock-autohost-service';

moduleForComponent('dashboards/live/host-mode/host-details', 'Integration | Component | dashboards/live/host-mode/host-details', {
  beforeEach() {
    this.user = Object.create({
      _id: 1,
      name: 'foo',
      displayName: 'FooBar',
      logo: 'logoPath'
    });

    this.noImageUser = Object.create({
      name: 'baz',
      displayName: 'BazFoo'
    });

    this.owner.register('service:autohost', MockAutohostService);
  }
});

test("it displays the host's display name", function (assert) {
  assert.expect(1);

  this.set('host', this.user);
  this.render(hbs`{{dashboards/live/host-mode/host-details host=host}}`);

  return wait().then(() => {
    assert.elementText(this.$('.js-host-details__display-name'), 'FooBar', 'uses the display name for the user');
  });
});

test("it displays the host's logo if they have one", function (assert) {
  assert.expect(2);

  this.set('host', this.user);
  this.render(hbs`{{dashboards/live/host-mode/host-details host=host}}`);

  return wait().then(() => {
    assert.equal(this.$('img').attr('src'), 'logoPath', 'sets the logo path as the img src');
    assert.equal(this.$('img').attr('title'), 'FooBar', 'sets the display name as the img title');
  });
});

test("it displays the default logo when the host has not set one", function (assert) {
  assert.expect(2);

  this.set('host', this.noImageUser);
  this.render(hbs`{{dashboards/live/host-mode/host-details host=host}}`);

  return wait().then(() => {
    assert.equal(
      this.$('img').attr('src'),
      'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_150x150.png',
      'sets the logo path as the img src'
    );
    assert.equal(this.$('img').attr('title'), 'BazFoo', 'sets the display name as the img title');
  });
});

test('not showing autohost add', function(assert) {
  assert.expect(4);

  this.set('host', Object.create({_id: 1, name: 'user1', displayName: 'User1', logo: 'url'}));
  this.render(hbs`{{dashboards/live/host-mode/host-details host=host shouldShowAutohostAdd=false}}`);

  return wait().then(() => {
    assert.equal(this.$('img').attr('src'), 'url', 'sets the logo path as the img src');
    assert.equal(this.$('img').attr('title'), 'User1', 'sets the display name as the img title');
    assert.equal(this.$('[data-test-selector=remove-button]').length, 0, 'remove button hidden');
    assert.equal(this.$('[data-test-selector=add-button]').length, 0, 'add button shows');
  });
});

test('add button', function(assert) {
  assert.expect(6);

  let autohost = this.owner.lookup('service:autohost');
  autohost.set('list', [{_id: 1, name: 'user2'}]);
  this.set('host', Object.create({_id: 2, name: 'user1', displayName: 'User1', logo: 'url'}));
  this.render(hbs`{{dashboards/live/host-mode/host-details host=host channel=host trackingSource='source' added=false}}`);

  return wait().then(() => {
    assert.equal(this.$('img').attr('src'), 'url', 'sets the logo path as the img src');
    assert.equal(this.$('img').attr('title'), 'User1', 'sets the display name as the img title');
    assert.equal(this.$('[data-test-selector=remove-button]').length, 0, 'remove button hidden');
    assert.equal(this.$('[data-test-selector=add-button]').length, 1, 'add button shows');
    this.$('[data-test-selector=add-button]').click();
    return wait();
  }).then(() => {
    assert.trackEvent('autohost_channel_update', { action: 'added', location: 'dashboard', source: 'source', target_channel: 'user1' });
    assert.equal(autohost.get('list').length, 2, 'user added');
  });
});

test('remove button', function(assert) {
  assert.expect(6);

  let autohost = this.owner.lookup('service:autohost');
  autohost.set('list', [{_id: 1, name: 'user1'}]);
  this.set('host', Object.create({_id: 1, name: 'user1', displayName: 'User1', logo: 'url'}));
  this.render(hbs`{{dashboards/live/host-mode/host-details host=host channel=host trackingSource='source' added=true}}`);

  return wait().then(() => {
    assert.equal(this.$('img').attr('src'), 'url', 'sets the logo path as the img src');
    assert.equal(this.$('img').attr('title'), 'User1', 'sets the display name as the img title');
    assert.equal(this.$('[data-test-selector=remove-button]').length, 1, 'remove button hidden');
    assert.equal(this.$('[data-test-selector=add-button]').length, 0, 'add button shows');
    this.$('[data-test-selector=remove-button]').click();
    return wait();
  }).then(() => {
    assert.trackEvent('autohost_channel_update', { action: 'removed', location: 'dashboard', source: 'source', target_channel: 'user1' });
    assert.equal(autohost.get('list').length, 0, 'user removed');
  });
});


