import { test } from 'ember-qunit';
import $ from 'jquery';
import Service from 'ember-service';
import RSVP from 'rsvp';
import hbs from 'htmlbars-inline-precompile';
import moduleForComponent from 'web-client/tests/helpers/module-for-component';

import { BadgesPage } from 'web-client/tests/pages/dashboards/subscription-dash';

const BADGE = {
  id: 1,
  required_tenure_months: 1,
  image_1x_url: "#test1",
  image_2x_url: "#test2",
  image_4x_url: "#test4",
  title: "Test Badge Title"
};

const CHANNEL_ID = 9000;

let subscriptionsStub;

moduleForComponent('dashboards/subscription-dash/badges/delete-form', 'Integration | Component | dashboards/subscription-dash/badges/delete form', {
  integration: true,

  beforeEach() {
    this.set('badge', BADGE);
    this.set('onBadgeDelete', () => {});
    this.set('channelID', CHANNEL_ID);

    this.$().append('<div id="wormhole-overlays"></div>');

    this.page = BadgesPage.create();
  }
});

test('it renders a valid form with badge images', function (assert) {
  assert.expect(6);

  this.render(hbs`
    {{dashboards/subscription-dash/badges/delete-form
      badge=badge
      onBadgeDelete=onBadgeDelete
      channelID=channelID}}
  `);

  let $images = this.$(`${this.page.form()} img`);

  assert.equal($images.eq(0).attr('src'), BADGE.image_1x_url, 'first <img> is the 1x badge image URL');
  assert.equal($images.eq(0).attr('alt'), BADGE.title, 'first <img> has the "alt" attribute set to the title');

  assert.equal($images.eq(1).attr('src'), BADGE.image_2x_url, 'second <img> is the 2x badge image URL');
  assert.equal($images.eq(1).attr('alt'), BADGE.title, 'second <img> has the "alt" attribute set to the title');

  assert.equal($images.eq(2).attr('src'), BADGE.image_4x_url, 'third <img> is the 4x badge image URL');
  assert.equal($images.eq(2).attr('alt'), BADGE.title, 'third <img> has the "alt" attribute set to the title');
});

test('it renders a confirmation modal on button[type="submit"] click', function (assert) {
  assert.expect(8);

  this.render(hbs`
    {{dashboards/subscription-dash/badges/delete-form
      badge=badge
      onBadgeDelete=onBadgeDelete
      channelID=channelID}}
  `);

  let $submitButton = this.$(this.page.formSubmitButton());

  assert.elementCount($(this.page.modal()), 0, 'modal is NOT rendered');

  $submitButton.click();

  assert.elementCount($(this.page.modal()), 1, 'modal is rendered');

  assert.equal(
    $(`${this.page.modal()} .item p`).text().trim(),
    'Deleting your subscriber badge could negatively impact your subscribers.',
    'modal paragraph text is rendered'
  );

  assert.ok($submitButton.prop('disabled'), 'submit button is disabled');
  assert.ok($submitButton.hasClass('button--disabled'), 'submit button has disabled class');

  $(this.page.modalCancelButton()).click();

  assert.elementCount($(this.page.modal()), 0, 'modal is once again NOT rendered');

  assert.notOk($submitButton.prop('disabled'), 'submit button is NOT disabled');
  assert.notOk($submitButton.hasClass('button--disabled'), 'submit button does NOT have disabled class');
});

test('confirming makes a call on Subscriptions service, invokes closure action, and renders a success notification', function (assert) {
  assert.expect(7);

  let onBadgeDelete = (deletedBadge) => {
    assert.ok(true, '"onBadgeDelete" closure action is invoked');
    assert.equal(deletedBadge.id, BADGE.id, 'badge is passed');
  };

  subscriptionsStub = Service.extend({
    deleteBadge({ channelID, badgeID }) {
      assert.equal(channelID, CHANNEL_ID, 'channel ID is correct');
      assert.equal(badgeID, BADGE.id, 'badge ID is correct');

      return RSVP.resolve(BADGE);
    }
  });

  this.set('onBadgeDelete', onBadgeDelete);
  this.register('service:subscriptions', subscriptionsStub);

  this.render(hbs`
    {{dashboards/subscription-dash/badges/delete-form
      badge=badge
      onBadgeDelete=onBadgeDelete
      channelID=channelID}}
  `);

  this.$(this.page.formSubmitButton()).click();

  assert.elementCount($(this.page.modal()), 1, 'modal is rendered');

  $(this.page.modalConfirmButton()).click();

  assert.elementCount($(this.page.modal()), 0, 'modal is NOT rendered');
  assert.didNotify('Subscriber badge successfully deleted');
});

test('failing to delete badge renders an error notification', function (assert) {
  assert.expect(3);

  subscriptionsStub = Service.extend({
    deleteBadge() {
      return RSVP.reject();
    }
  });

  this.register('service:subscriptions', subscriptionsStub);

  this.render(hbs`
    {{dashboards/subscription-dash/badges/delete-form
      badge=badge
      onBadgeDelete=onBadgeDelete
      channelID=channelID}}
  `);

  this.$(this.page.formSubmitButton()).click();

  assert.elementCount($(this.page.modal()), 1, 'modal is rendered');

  $(this.page.modalConfirmButton()).click();

  assert.elementCount($(this.page.modal()), 0, 'modal is NOT rendered');
  assert.didNotify('Failed to delete subscriber badge');
});
