import { test, skip } from 'ember-qunit';
import Service from 'ember-service';
import wait from 'ember-test-helpers/wait';
import hbs from 'htmlbars-inline-precompile';
import moduleForComponent from 'web-client/tests/helpers/module-for-component';
import {
  AuthenticatedSession,
  AuthenticatedSessionWithPrime,
  UnAuthenticatedSession
} from 'web-client/tests/helpers/mock-session-service';
import moment from 'moment';
import { loadGameDetail, loadOffer } from 'web-client/tests/integration/components/directory/game-details/game-detail-helper';

const GAME_FREE = {
  product: {
    title: "Free",
    game_id: "Free-id"
  }
};

const GAME_NOT_FREE = {
  product: {
    title: "Not Free"
  }
};

const GAME_TO_BUY = {
  actionDetails: {
    destination_url: "Url to twitch.amazon.com checkout"
  },
  product: {
    title: "Game to Buy",
    // This is evaluated when the test is imported... ensure it outlives the tests
    release_end_date: moment().add(20, 'minute').toISOString()
  }
};

const GAME_NOT_FOR_SALE = {
  product: {
    title: "Game not for sale",
    release_end_date: moment().subtract(1, 'minute').toISOString()
  }
};

const OFFER_VALID = {
  applicableGame: GAME_FREE.product.title,
  contentDeliveryMethod: "DIRECT_ENTITLEMENT"
};

const OFFER_EXPIRED = {
  applicableGame: "Tribbles",
  contentDeliveryMethod: "DIRECT_ENTITLEMENT",
  endTime: "2015-11-01T06:59:59Z"
};

moduleForComponent('directory/game-details/get-game-card/offer', 'Integration | Component | directory/game details/get game card/offer', {
  beforeEach() {
    this.store = this.owner.lookup('service:store');
    this.loadGameDetail = loadGameDetail;
    this.loadOffer = loadOffer;
  }
});


test('Not presented when not free with prime.', function(assert) {
  this.loadOffer(OFFER_VALID);
  this.loadGameDetail(GAME_NOT_FREE);

  this.render(hbs`{{directory/game-details/get-game-card/offer
    gameDetail=gameDetail
    offer=offer
  }}`);
  assert.elementText(this.$('[data-test-offer-detail]'), '', 'No Offers presented.');
});


test('Not presented when prime offer expired.', function(assert) {
  this.loadOffer(OFFER_EXPIRED);
  this.loadGameDetail(GAME_FREE);

  this.render(hbs`{{directory/game-details/get-game-card/offer
    gameDetail=gameDetail
    offer=offer
  }}`);
  assert.elementText(this.$('[data-test-offer-detail]'), '', 'No Offers presented.');
});

// TODO
skip('sign up for prime call to action', function(assert) {
  // Game is free with prime
  // User does not have prime
  // Prime is available in geo
  this.loadGameDetail(GAME_FREE);
  this.loadOffer(OFFER_VALID);
  this.register('service:session', AuthenticatedSession);
  this.register('service:premium-geo', Service.extend({
    showPrimeContent: true
  }));

  this.render(hbs`{{directory/game-details/get-game-card/offer
    gameDetail=gameDetail
    offer=offer
  }}`);
  return wait().then(() => {
    assert.elementText(this.$('[data-test-offer-detail]'), 'Start Your 30-day Free Trial', 'Start trial');
  });
});


// TODO
skip('buy call to action', function(assert) {
  this.loadGameDetail(GAME_TO_BUY);
  this.loadOffer(OFFER_VALID);
  this.register('service:session', AuthenticatedSession);
  this.register('service:premium-geo', Service.extend({
    showPrimeContent: true
  }));

  this.render(hbs`{{directory/game-details/get-game-card/offer
    gameDetail=gameDetail
    offer=offer
  }}`);
  return wait().then(() => {
    assert.elementText(this.$('[data-test-offer-detail-button]'), 'Buy this Game', 'Buy This Game shown');
  });

});


test('buy call to action not shown when not for sale', function(assert) {
  this.loadGameDetail(GAME_NOT_FOR_SALE);
  this.loadOffer(OFFER_VALID);
  this.register('service:session', AuthenticatedSession);
  this.register('service:premium-geo', Service.extend({
    showPrimeContent: true
  }));

  this.render(hbs`{{directory/game-details/get-game-card/offer
    gameDetail=gameDetail
    offer=offer
  }}`);
  return wait().then(() => {
    assert.elementText(this.$('[data-test-offer-detail-button]'), '', 'Buy This Game NOT shown');
  });


});

// TODO
skip('claim your offer call to action', function(assert) {
  // Game is free with prime
  // User does have prime
  // Prime is available in geo
  this.loadGameDetail(GAME_FREE);
  this.loadOffer(OFFER_VALID);
  this.register('service:session', AuthenticatedSessionWithPrime);
  this.register('service:premium-geo', Service.extend({
    showPrimeContent: true
  }));

  this.render(hbs`{{directory/game-details/get-game-card/offer
    gameDetail=gameDetail
    offer=offer
  }}`);

  return wait().then(() => {
    assert.elementText(this.$('[data-test-offer-detail]'), 'Get this game with Prime', 'Claim offer with twitch prime');
  });
});


// TODO
skip('can buy the game when logged in', function(assert) {
  assert.expect(2);
  this.loadGameDetail(GAME_TO_BUY);
  this.loadOffer(OFFER_VALID);
  this.register('service:session', AuthenticatedSessionWithPrime);
  this.register('service:premium-geo', Service.extend({
    showPrimeContent: true
  }));

  let checkoutUrl = null;
  let windowOpenCount = 0;
  // We stub out windowOpen to prevent pop-out and verify it was called correctly.
  let windowOpen = (url) => {
    checkoutUrl = url;
    windowOpenCount++;
  };
  this.set('windowOpen', windowOpen);

  this.render(hbs`{{directory/game-details/get-game-card/offer
    gameDetail=gameDetail
    offer=offer
    windowOpen=windowOpen
  }}`);


  this.$('[data-test-offer-detail-button]').click();

  return wait().then(() => {
    assert.equal(windowOpenCount, 1, 'Window opened once');
    assert.equal(checkoutUrl, GAME_TO_BUY.actionDetails.destination_url, 'Checkout window opened at correct URL');
  });
});

// TODO
skip('can buy the game when logged out', function(assert) {
  assert.expect(2);
  this.loadGameDetail(GAME_TO_BUY);
  this.loadOffer(OFFER_VALID);
  this.register('service:session', UnAuthenticatedSession);
  this.register('service:premium-geo', Service.extend({
    showPrimeContent: true
  }));

  let checkoutUrl = null;
  let windowOpenCount = 0;
  // We stub out windowOpen to prevent pop-out and verify it was called correctly.
  let windowOpen = (url) => {
    checkoutUrl = url;
    windowOpenCount++;
  };
  this.set('windowOpen', windowOpen);

  this.render(hbs`{{directory/game-details/get-game-card/offer
    gameDetail=gameDetail
    offer=offer
    windowOpen=windowOpen
  }}`);


  this.$('[data-test-offer-detail-button]').click();

  return wait().then(() => {
    assert.equal(windowOpenCount, 1, 'Window opened once');
    assert.equal(checkoutUrl, GAME_TO_BUY.actionDetails.destination_url, 'Checkout window opened at correct URL');
  });
});

// TODO
skip('can use twitch prime to buy the game', function(assert) {
  assert.expect(3);

  this.loadGameDetail(GAME_FREE);
  this.loadOffer(OFFER_VALID);
  this.register('service:session', AuthenticatedSessionWithPrime);
  this.register('service:premium-geo', Service.extend({
    showPrimeContent: true
  }));

  let requestCalls = [];
  this.server.post('http://api.twitch.tv/api/premium/offers/:offer_id/claim', function(schema, request) {
    requestCalls.push(request);
    return {
      type: 'DIRECT_ENTITLEMENT',
      claimData: 'fuel-is-going-to-be-great'
    };
  });

  let locationRedirect = function(href) {
    assert.equal(href, `/directory/game/${GAME_FREE.product.game_id}/details?redeemed=true`, 'redirects to the claim url');
  };

  this.set('locationRedirect', locationRedirect);

  this.render(hbs`{{directory/game-details/get-game-card/offer
    gameDetail=gameDetail
    offer=offer
    locationRedirect=locationRedirect
  }}`);

  return wait().then(() => {
    assert.equal(this.$('[data-test-offer-detail]').length, 1, "Prime button rendered");
    this.$('[data-test-offer-detail]').click();
    return wait().then(() => {
      assert.equal(requestCalls.length, 1, 'The api service is called once.');
    });
  });
});
