import { test } from 'ember-qunit';
import hbs from 'htmlbars-inline-precompile';
import EmberObject from 'ember-object';
import run from 'ember-runloop';
import Service from 'ember-service';
import moduleForComponent from 'web-client/tests/helpers/module-for-component';
import videoAppealScenario from 'web-client/mirage/scenarios/video-appeal';

moduleForComponent('manager/video-card', 'Integration | Component | manager/video-card', {
  beforeEach() {
    let video = EmberObject.create();

    this.set('video', video);
    this.setVideoProperties = (attrs) => {
      run(() => video.setProperties(attrs));
    };

    this.owner.register('service:session', Service.extend({
      userData: {
        login: 'username'
      }
    }));

    this.template = hbs`
      {{manager/video-card video=video}}
    `;
  }
});

test('when video is "new", is in uploading state', function(assert) {
  this.setVideoProperties({
    status: 'new',
    videoUpload: null
  });

  this.render(this.template);

  assert.equal(currentStatus(this.$()), 'uploading');
});

test('when video is "created" with no upload, is in error state', function(assert) {
  this.setVideoProperties({ status: 'created' });
  this.render(this.template);

  assert.equal(currentStatus(this.$()), 'error');
});

test('when video is "created" with a failed upload, is in error state', function(assert) {
  this.setVideoProperties({
    status: 'created',
    videoUpload: EmberObject.create({
      isError: true,
      isComplete: true
    })
  });

  this.render(this.template);

  assert.equal(currentStatus(this.$()), 'error');
});

test('when video is "created" with pending upload, is in uploading state', function(assert) {
  this.setVideoProperties({
    status: 'created',
    videoUpload: EmberObject.create({
      isUploading: true,
      isComplete: false
    })
  });

  this.render(this.template);

  assert.equal(currentStatus(this.$()), 'uploading');
});

test('when video is "created" with unstarted upload, is in queued state', function(assert) {
  this.setVideoProperties({
    status: 'created',
    videoUpload: EmberObject.create({
      isUploading: false,
      isComplete: false
    })
  });

  this.render(this.template);

  assert.equal(currentStatus(this.$()), 'queued');
});

test('when video is "created" with completed upload, is in transcoding state', function(assert) {
  this.setVideoProperties({
    status: 'created',
    videoUpload: EmberObject.create({
      isComplete: true,
      isSuccess: true
    })
  });
  this.render(this.template);
  assert.equal(currentStatus(this.$()), 'transcoding');
});

test('when video is "pending_transcoding", is in transcoding state', function(assert) {
  this.setVideoProperties({
    status: 'transcoding'
  });

  this.render(this.template);

  assert.equal(currentStatus(this.$()), 'transcoding');
});

test('when video is "transcoding", is in transcoding state', function(assert) {
  this.setVideoProperties({
    status: 'transcoding'
  });

  this.render(this.template);

  assert.equal(currentStatus(this.$()), 'transcoding');
});

test('when video is "uploading" with no videoUpload, is in error state', function(assert) {
  this.setVideoProperties({
    status: 'uploading'
  });

  this.render(this.template);

  assert.equal(currentStatus(this.$()), 'error');
});

test('when video is "failed", is in error state', function(assert) {
  this.setVideoProperties({
    status: 'failed'
  });

  this.render(this.template);

  assert.equal(currentStatus(this.$()), 'error');
});

test('when video is "recorded", is in processed state', function(assert) {
  this.setVideoProperties({
    status: 'recorded'
  });

  this.render(this.template);

  assert.equal(currentStatus(this.$()), 'processed');
});

test('when video status is "new" and video has isError property, is in error state', function(assert) {
  this.setVideoProperties({
    status: 'new',
    isError: true
  });

  this.render(this.template);

  assert.equal(currentStatus(this.$()), 'error');
});

test('when the video is complete, the actions are visible', function(assert) {
  this.setVideoProperties({
    status: 'recorded'
  });

  this.render(this.template);

  assert.equal(this.$('[data-test-actions]').css('visibility'), 'visible', 'actions are visible');
});

test('when video has errored, the actions are hidden', function(assert) {
  this.setVideoProperties({
    status: 'failed'
  });

  this.render(this.template);

  assert.equal(this.$('[data-test-actions]').css('visibility'), 'hidden', 'actions are hidden');
});

test('when video is complete and scheduled', function(assert) {
  this.setVideoProperties({
    status: 'recorded',
    isScheduled: true
  });

  this.render(this.template);

  assert.equal(this.$('[data-test-scheduled-overlay]').length, 1, 'shows scheduled status');
});

test('when video is complete and published', function(assert) {
  this.setVideoProperties({
    status: 'recorded',
    isPublished: true,
    views: 1000
  });

  this.render(this.template);

  assert.stringIncludes(this.$('[data-test-views]').text(), '1,000', 'shows view count');
});

test('when video is complete and unpublished', function(assert) {
  this.setVideoProperties({
    status: 'recorded',
    isUnpublished: true,
    views: 1000
  });

  this.render(this.template);

  assert.stringIncludes(this.$('[data-test-views]').text(), '1,000', 'shows view count');
  assert.stringIncludes(this.$('[data-test-badge]').text(), 'Unpublished', 'shows unpublished badge');
});

test('when public video has a muted track', function(assert) {
  let videoId = 'v12345';

  videoAppealScenario(this.server, videoId);
  this.setVideoProperties({
    id: videoId,
    status: 'recorded',
    isPublished: true,
    isMuted: true
  });

  this.render(this.template);

  assert.equal(this.$('[data-test-mute]').length, 1, 'shows mute svg icon');
  assert.equal(this.$('[data-test-mute]').hasClass('card__meta--alert'), true, 'shows mute svg icon below unpublished badge');
});

test('when private video has a muted track', function(assert) {
  let videoId = 'v12345';
  videoAppealScenario(this.server, videoId);

  this.setVideoProperties({
    id: videoId,
    status: 'recorded',
    isUnpublished: true,
    isMuted: true
  });

  this.render(this.template);

  assert.stringIncludes(this.$('[data-test-badge]').text(), 'Unpublished', 'shows unpublished badge');
  assert.equal(this.$('[data-test-mute]').hasClass('card__meta--below'), true, 'shows mute svg icon below unpublished badge');
});

function currentStatus(element) {
  return element.find('[data-status]').attr('data-status');
}

test('when loggined in user is owner of vod', function(assert) {
  let videoId = 'v12345';

  videoAppealScenario(this.server, videoId);
  this.setVideoProperties({
    id: videoId,
    status: 'recorded',
    isPublished: true,
    isMuted: true,
    channelName: "username"
  });

  this.render(this.template);
  let loggedInUser = this.owner.lookup('service:session').get('userData').login;

  assert.equal(loggedInUser, this.video.channelName, 'can retrieve vod appeals');
});
