/* globals Twitch */
import moduleForComponent from 'web-client/tests/helpers/module-for-component';
import { test } from 'ember-qunit';
import hbs from 'htmlbars-inline-precompile';
import Service from 'ember-service';
import EmberObject from 'ember-object';
import FakePlayer from 'web-client/tests/helpers/fake-player';
import MockSession from 'web-client/tests/helpers/mock-session-service';
import MockPersistentPlayer from 'web-client/tests/helpers/mock-persistent-player-service';
import MockPersistentPlayerContent from 'web-client/tests/helpers/mock-persistent-player-content-service';
import sinon from 'sinon';

const MockLayout = Service.extend({
  on: sinon.stub(),
  off: sinon.stub(),
  setTheatreMode: sinon.stub(),
  isTheatreMode: false,

  fullSizePlayerDimensions: {
    height: 600,
    width: 900,
    targetWidth: 900
  }
});

moduleForComponent('Integration | Component | persistent-player', {
  integration: true,
  beforeEach() {
    this.mockChannel = EmberObject.create({
      id: 'twitch',
      name: 'twitch',
      displayName: 'twitch'
    });

    this.register('service:persistentPlayer', MockPersistentPlayer);
    this.register('service:persistentPlayerContent', MockPersistentPlayerContent);
    this.register('service:layout', MockLayout);
    this.register('service:session', MockSession);

    this.inject.service('persistentPlayer', { as: 'persistentPlayer' });
    this.inject.service('persistentPlayerContent', { as: 'persistentPlayerContent' });
    this.inject.service('layout', { as: 'layout' });

    Twitch.video.Player = FakePlayer;

    this.set('persistentPlayerContent.currentChannel', this.mockChannel);
    this.set('persistentPlayerContent.targetChannel', this.mockChannel);
  },

  afterEach() {
    Twitch.video.Player = undefined;
  }
});

test('when isVisible is false', function(assert) {
  this.set('persistentPlayer.isVisible', false);
  this.render(hbs`{{persistent-player}}`);
  let playerElement = this.$().children(0);

  assert.ok(playerElement.hasClass('hidden'), 'the container element is hidden');
  assert.equal(playerElement.find('.dynamic-player').length, 0, 'the player element is not rendered');
});

test('when isMini is false', function(assert) {
  this.set('persistentPlayer.isMini', false);
  this.set('persistentPlayer.isVisible', true);
  this.render(hbs`{{persistent-player}}`);
  let playerElement = this.$().children(0);

  assert.equal(playerElement.find('.dynamic-player').length, 1, 'the player element is rendered');
  assert.equal(playerElement.find('.player-mini__titlebar').length, 0, 'the titlebar element is not rendered');
  assert.notOk(playerElement.hasClass('player-mini'), 'the mini class is not present');

  let expectedHeight = `${this.get('layout.fullSizePlayerDimensions.height')}px`;
  let expectedWidth = `${this.get('layout.fullSizePlayerDimensions.width')}px`;
  let expectedTop = `${this.get('persistentPlayer.fullSizePlayerLocation.top')}px`;

  assert.equal(playerElement.css('height'), expectedHeight, 'should have the specified height');
  assert.equal(playerElement.css('width'), expectedWidth, 'should have the specified width');
  assert.equal(playerElement.css('top'), expectedTop, 'should have the specified top');
});

test('when isMini is true', function(assert) {
  this.set('persistentPlayer.isMini', true);
  this.set('persistentPlayer.isVisible', true);
  this.render(hbs`{{persistent-player}}`);
  let playerElement = this.$().children(0);

  assert.equal(playerElement.find('.dynamic-player').length, 1, 'the player element is rendered');
  assert.equal(playerElement.find('.player-mini__titlebar').length, 1, 'the titlebar element is rendered');
  assert.ok(playerElement.hasClass('player-mini'), 'the mini class is present');
});

test('when isPreAnimation is true', function(assert) {
  this.set('persistentPlayer.isMini', false);
  this.set('persistentPlayer.isPreAnimation', true);
  this.set('persistentPlayer.isVisible', true);

  this.render(hbs`{{persistent-player}}`);
  let playerElement = this.$().children(0);

  assert.equal(playerElement.find('.dynamic-player').length, 1, 'the player element is rendered');
  assert.equal(playerElement.find('.player-mini__titlebar').length, 0, 'the titlebar element is not rendered');
  assert.notOk(playerElement.hasClass('player-mini'), 'the mini class is not present');

  let expectedHeight = `${this.get('persistentPlayer.playerDimensions.height')}px`;
  let expectedWidth = `${this.get('persistentPlayer.playerDimensions.width')}px`;
  let expectedTop = `${this.get('persistentPlayer.animationCoords.originX')}px`;
  let expectedLeft = `${this.get('persistentPlayer.animationCoords.originY')}px`;

  assert.equal(playerElement.css('height'), expectedHeight, 'should have the specified height');
  assert.equal(playerElement.css('width'), expectedWidth, 'should have the specified width');
  assert.equal(playerElement.css('top'), expectedTop, 'should have the specified top');
  assert.equal(playerElement.css('left'), expectedLeft, 'should have the specified left');
});

test('when isAnimating is true', function(assert) {
  this.set('persistentPlayer.isMini', false);
  this.set('persistentPlayer.isAnimating', true);
  this.set('persistentPlayer.isVisible', true);

  this.render(hbs`{{persistent-player}}`);
  let playerElement = this.$().children(0);

  assert.equal(playerElement.find('.dynamic-player').length, 1, 'the player element is rendered');
  assert.equal(playerElement.find('.player-mini__titlebar').length, 0, 'the titlebar element is not rendered');
  assert.notOk(playerElement.hasClass('player-mini'), 'the mini class is not present');

  let expectedHeight = `${this.get('persistentPlayer.playerDimensions.height')}px`;
  let expectedWidth = `${this.get('persistentPlayer.playerDimensions.width')}px`;
  let expectedTop = `${this.get('persistentPlayer.animationCoords.originX')}px`;
  let expectedLeft = `${this.get('persistentPlayer.animationCoords.originY')}px`;


  assert.equal(playerElement.css('height'), expectedHeight, 'should have the specified height');
  assert.equal(playerElement.css('width'), expectedWidth, 'should have the specified width');
  assert.equal(playerElement.css('top'), expectedTop, 'should have the specified top');
  assert.equal(playerElement.css('left'), expectedLeft, 'should have the specified left');

  assert.ok(playerElement.css('transition'), 'should have a transition property set');

 /**
  * We would like to be able to test that the transform property is set, but phantomJS
  * won't let us. Uncomment this when we get a better test browser on Jenkins

  let expectedTranslateX = `${this.get('persistentPlayer.animationCoords.translateX')}`;
  let expectedTranslateY = `${this.get('persistentPlayer.animationCoords.translateY')}`;
  let expectedScaleX = `${this.get('persistentPlayer.animationCoords.scaleX')}`;
  let expectedScaleY = `${this.get('persistentPlayer.animationCoords.scaleY')}`;

  let expectedTransform = `matrix(${expectedScaleX}, 0, 0, ${expectedScaleY}, ${expectedTranslateX}, ${expectedTranslateY})`;
  assert.equal(playerElement.css('transform'), expectedTransform, 'should have the specified transform');
  */
});

test('when the content is a live channel', function(assert) {
  this.set('persistentPlayer.isMini', true);
  this.set('persistentPlayer.isVisible', true);
  this.mockChannel.set('playerIsLive', true);

  this.render(hbs`{{persistent-player}}`);

  assert.elementText('.js-player-mini__titlebar', 'Watching twitch', 'should say "Watching" the current channel');
});

test('when the content is a hosted channel', function(assert) {
  let mockHostedChannel = EmberObject.create({
    id: 'foilstormcrow',
    name: 'foilstormcrow',
    displayName: 'foilstormcrow'
  });

  this.set('persistentPlayer.isMini', true);
  this.set('persistentPlayer.isVisible', true);
  this.set('persistentPlayerContent.isHosted', true);
  this.set('persistentPlayerContent.targetChannel', mockHostedChannel);

  this.render(hbs`{{persistent-player}}`);

  assert.elementText('.js-player-mini__titlebar', 'Hosting foilstormcrow', 'should say "Hosting", the current channel');
});

test('when the content is a VOD', function(assert) {
  let mockVOD = EmberObject.create({
    id: 12345,
    owner: this.mockChannel
  });

  this.set('persistentPlayer.isMini', true);
  this.set('persistentPlayer.isVisible', true);
  this.set('persistentPlayerContent.currentVOD', mockVOD);

  this.render(hbs`{{persistent-player}}`);

  assert.elementText('.js-player-mini__titlebar', 'Watching twitch', 'should say "Watching" the VOD owner channel');
});
