import moduleForComponent from 'web-client/tests/helpers/module-for-component';
import { test } from 'ember-qunit';
import sinonTest from 'ember-sinon-qunit/test-support/test';
import hbs from 'htmlbars-inline-precompile';
import FakePlayer from '../../helpers/fake-player';
import RSVP from 'rsvp';
import Mousetrap from 'mousetrap';
import dispatchKeyEvent from '../../helpers/dispatch-key-event';

const KEY_X = 88;

moduleForComponent('twitch-player2', 'Integration | Component | twitch-player2', {
  beforeEach() {
    this.set('Player', FakePlayer);
    this.on('stashPlayer', (playerComponent) => {
      this.player = playerComponent.get('player');
    });

    this.set('canGenerateClips', false);
  }
});

test('toggles theatre mode with `alt+t`', function(assert) {
  let selector = '[data-test-selector=twitch-player-wrapper]';

  this.render(hbs`{{twitch-player2 canGenerateClips=canGenerateClips video='v123' Player=Player}}`);
  assert.ok(this.$(selector).hasClass('dynamic-player'));
  Mousetrap.trigger('alt+t');
  assert.ok(!this.$(selector).hasClass('dynamic-player'));
  Mousetrap.trigger('alt+t');
  assert.ok(this.$(selector).hasClass('dynamic-player'));
});

test('closes theatre mode with `esc`', function(assert) {
  let selector = '[data-test-selector=twitch-player-wrapper]';

  this.render(hbs`{{twitch-player2
     canGenerateClips=canGenerateClips video='v123' Player=Player}}`);
  assert.ok(this.$(selector).hasClass('dynamic-player'));
  Mousetrap.trigger('alt+t');
  assert.ok(!this.$(selector).hasClass('dynamic-player'));
  Mousetrap.trigger('esc');
  assert.ok(this.$(selector).hasClass('dynamic-player'));
});

test('it sets a channel on the player', function(assert) {
  this.set('channel', {
    id: 'pianoimproman'
  });

  this.render(hbs`
    {{twitch-player2
       canGenerateClips=canGenerateClips
       channel=channel
       Player=Player
       onPlayerInit=(action 'stashPlayer')}}
  `);

  assert.equal(this.player.channel, 'pianoimproman');
});

test('it sets a video on the player', function(assert) {
  this.render(hbs`
    {{twitch-player2
       canGenerateClips=canGenerateClips
       video='v123'
       Player=Player
       onPlayerInit=(action 'stashPlayer')}}
  `);

  assert.equal(this.player.video, 'v123');
});

test('changing the channel', function(assert) {
  this.set('channel', { id: 'pianoimproman' });

  this.render(hbs`
    {{twitch-player2
       canGenerateClips=canGenerateClips
       channel=channel
       Player=Player
       onPlayerInit=(action 'stashPlayer')}}
  `);

  this.set('channel', { id: 'mitchlloyd' });

  assert.equal(this.player.channel, 'mitchlloyd');
});

test('changing the video', function(assert) {
  this.set('video', 'v111');

  this.render(hbs`
    {{twitch-player2
       canGenerateClips=canGenerateClips
       video=video
       Player=Player
       onPlayerInit=(action 'stashPlayer')}}
  `);

  this.set('video', 'v222');

  assert.equal(this.player.video, 'v222');
});

test('can generate clips when canGenerateClips is true', function (assert) {
  this.set('channel', { id: 'pianoimproman' });
  this.set('canGenerateClips', false);

  this.render(hbs`
    {{twitch-player2
       canGenerateClips=canGenerateClips
       channel=channel
       Player=Player
       canGenerateClips=canGenerateClips
       onPlayerInit=(action 'stashPlayer')}}
  `);

  let keyEventOptions = { altKey: true, which: KEY_X };
  dispatchKeyEvent('keydown', keyEventOptions);
  assert.ok(!this.player.clipRecorded, 'Clip not recorded while canGenerateClips is false');

  this.set('canGenerateClips', true);
  dispatchKeyEvent('keydown', keyEventOptions);
  assert.ok(this.player.clipRecorded, 'Clip recorded when canGenerateClips is true');
});

sinonTest("handling player's transitiontovod event", function(assert) {
  let router = this.owner.lookup('router:main');
  let transitionTo = this.stub(router, 'transitionTo').returns(RSVP.resolve());

  this.render(hbs`
    {{twitch-player2
       canGenerateClips=canGenerateClips
       Player=Player
       onPlayerInit=(action 'stashPlayer')}}
  `);

  this.player.trigger('transitiontovod', [{
    _id: 'v111',
    channel: {
      name: 'twitch'
    }
  }]);

  assert.ok(transitionTo.calledWith('videos', '111', {queryParams: {}}), 'transitions to a video');
});

test('handling player contentShowing event', function(assert) {
  assert.expect(1);

  this.on('verifyCall', function() {
    assert.ok(true, 'videoPlayerContentShowing sent');
  });

  this.render(hbs`
    {{twitch-player2
        Player=Player
        canGenerateClips=false
        onPlayerInit=(action 'stashPlayer')
        videoPlayerContentShowing=(action 'verifyCall')
    }}
  `);

  this.player.trigger('contentShowing');
});

test('call setTrackingProperties when changing to host mode', function(assert) {
  let hostedChannel = { id: 'monstercat', name: 'monstercat' };
  let channel = { id: 'extalix', name: 'extalix' };
  let hostingChannel = { id: 'extalix', name: 'extalix', hostModeTarget: hostedChannel };
  this.set('channel', channel);

  this.render(hbs`
    {{twitch-player2
      canGenerateClips=canGenerateClips
      channel=channel
      hostChannel=channel
      Player=Player
      onPlayerInit=(action 'stashPlayer')}}
    }}
  `);

  assert.deepEqual(this.player.trackingProperties, {});

  this.set('channel', hostingChannel);

  assert.deepEqual(this.player.trackingProperties, {
    host_channel: 'extalix'
  });
});

test('when in theatre mode and mini UI is enabled', function(assert) {
  this.set('hasMiniControls', false);

  this.render(hbs`
    {{twitch-player2
      canGenerateClips=false
      Player=Player
      playerIsPersistent=true
      hasMiniControls=hasMiniControls
      onPlayerInit=(action 'stashPlayer')
    }}
  `);

  this.player.setTheatre(true);
  this.set('hasMiniControls', true);

  assert.notOk(this.player.getTheatre(), 'removes theatre mode');
});
