import { test } from 'ember-qunit';
import wait from 'ember-test-helpers/wait';
import hbs from 'htmlbars-inline-precompile';
import moduleForComponent from 'web-client/tests/helpers/module-for-component';

import Service from 'ember-service';
import moment from 'moment';

const AuthenticatedSession = Service.extend({
  isAuthenticated: true,
  userData: {
    login: 'kiwi',
    id: '123'
  }
});

moduleForComponent('twitch-prime/offer-list', 'Integration | Component | twitch-prime/offer-list', {

  beforeEach() {
    this.register('service:session', AuthenticatedSession);
  }
});

test('it has a title', function(assert) {

  this.render(hbs`
    {{twitch-prime/offer-list}}
  `);

  return wait().then(() => {
    assert.elementText(this.$('.js-offer-list-title'), "Free with Prime");
  });
});

test('renders list of offers', function(assert) {
  let offersCount = 3;
  this.server.createList('offer', offersCount);

  this.render(hbs`
    {{#twitch-prime/offer-list as |navOffers hasPrime|}}
      {{#each navOffers as |offer|}}
        {{twitch-prime/offer-item offer=offer}}
      {{/each}}
    {{/twitch-prime/offer-list}}
  `);

  return wait().then(() => {
    assert.elementCount(this.$('.offer-item'), offersCount);
  });
});

test('renders list of limited number offers', function(assert) {
  let offersCount = 4;
  let expectedCount = 3;
  this.server.createList('offer', offersCount);

  this.render(hbs`
    {{#twitch-prime/offer-list as |navOffers hasPrime|}}
      {{#each navOffers as |offer|}}
        {{twitch-prime/offer-item offer=offer}}
      {{/each}}
    {{/twitch-prime/offer-list}}
  `);

  return wait().then(() => {
    assert.elementCount(this.$('.offer-item'), expectedCount);
  });
});

test('renders list of all offers if showAllOffers is true', function(assert) {
  let offersCount = 4;
  this.server.createList('offer', offersCount);

  this.render(hbs`
    {{#twitch-prime/offer-list showAllOffers=true as |navOffers hasPrime|}}
      {{#each navOffers as |offer|}}
        {{twitch-prime/offer-item offer=offer}}
      {{/each}}
    {{/twitch-prime/offer-list}}
  `);

  return wait().then(() => {
    assert.elementCount(this.$('.offer-item'), offersCount);
  });
});

test('renders only a limited number of offers if the left nav is minimized even if showAllOffers is true', function(assert) {
  let offersCount = 4;
  let expectedCount = 3;
  this.server.createList('offer', offersCount);

  this.render(hbs`
    {{#twitch-prime/offer-list showAllOffers=true isLeftColumnClosed=true as |navOffers hasPrime|}}
      {{#each navOffers as |offer|}}
        {{twitch-prime/offer-item offer=offer}}
      {{/each}}
    {{/twitch-prime/offer-list}}
  `);

  return wait().then(() => {
    assert.elementCount(this.$('.offer-item'), expectedCount);
  });
});

test('do not display expired offers', function(assert){
  assert.expect(1);

  this.server.create('offer', {
    endTime: moment(new Date()).subtract(3, 'minutes').format()
  });

  this.render(hbs`
    {{#twitch-prime/offer-list as |navOffers hasPrime showAllOffers|}}
      {{#each navOffers as |offer|}}
        {{twitch-prime/offer-item offer=offer}}
      {{/each}}
    {{/twitch-prime/offer-list}}
  `);

  return wait().then(() => {
    assert.elementCount(this.$('.warp__item--thumbnail'), 0, 'Expired offer is not shown');
  });
}),

test('renders claimed offers after unclaimed offers in showAllOffers mode', function(assert) {
  assert.expect(4);

  let offersCount = 4;
  let expectedCount = 2;
  this.server.createList('offer', offersCount);

  let offerIds = [this.server.db.offers[1].offerId, this.server.db.offers[3].offerId];
  let claimedOfferIds = offerIds.map((offerId) => {
    return `${offerId}-123`;
  });
  let claimedOfferTitles = [this.server.db.offers[1].offerTitle, this.server.db.offers[3].offerTitle];

  this.set('claimedOfferIds', claimedOfferIds);

  this.render(hbs`
    {{#twitch-prime/offer-list showAllOffers=true claimedOfferIds=claimedOfferIds as |navOffers hasPrime showAllOffers hiddenOfferIds claimedOfferIds|}}
      {{#each navOffers as |offer|}}
        {{twitch-prime/offer-item offer=offer showAllOffers=showAllOffers claimedOfferIds=claimedOfferIds}}
      {{/each}}
    {{/twitch-prime/offer-list}}
  `);

  return wait().then(() => {
    assert.elementCount(this.$('.offer-item'), offersCount);
    assert.elementCount(this.$('.warp__item--thumbnail'), expectedCount);
    assert.elementText(this.$('.offer-item .warp__title')[2], claimedOfferTitles[0]);
    assert.elementText(this.$('.offer-item .warp__title')[3], claimedOfferTitles[1]);
  });
});

test('can hide offers if the left nav is open and not in showAllOffers mode', function(assert) {
  assert.expect(1);

  let offersCount = 4;
  let expectedCount = 3;
  this.server.createList('offer', offersCount);

  this.render(hbs`
    {{#twitch-prime/offer-list as |navOffers hasPrime showAllOffers hiddenOfferIds claimedOfferIds|}}
      {{#each navOffers as |offer|}}
        {{twitch-prime/offer-item offer=offer hiddenOfferIds=hiddenOfferIds}}
      {{/each}}
    {{/twitch-prime/offer-list}}
  `);

  this.$('.offer-item__hide-offer').first().click();

  return wait().then(() => {
    assert.elementCount(this.$('.offer-item'), expectedCount);
  });
});

test('renders hidden offers in showAllOffers mode', function(assert) {
  assert.expect(1);

  let offersCount = 4;
  let expectedCount = 4;
  this.server.createList('offer', offersCount);

  let hiddenOfferIds = [this.server.db.offers[0].offerId];

  this.set('hiddenOfferIds', hiddenOfferIds);

  this.render(hbs`
    {{#twitch-prime/offer-list showAllOffers=true hiddenOfferIds=hiddenOfferIds as |navOffers hasPrime showAllOffers hiddenOfferIds claimedOfferIds|}}
      {{#each navOffers as |offer|}}
        {{twitch-prime/offer-item offer=offer showAllOffers=showAllOffers hiddenOfferIds=hiddenOfferIds}}
      {{/each}}
    {{/twitch-prime/offer-list}}
  `);

  return wait().then(() => {
    assert.elementCount(this.$('.offer-item'), expectedCount);
  });
});

test('does not render hidden offers in collapsed left nav mode', function(assert) {
  assert.expect(1);

  let offersCount = 4;
  let expectedCount = 3;
  this.server.createList('offer', offersCount);

  let hiddenOfferIds = [this.server.db.offers[0].offerId];

  this.set('hiddenOfferIds', hiddenOfferIds);

  this.render(hbs`
    {{#twitch-prime/offer-list isLeftColumnClosed=true hiddenOfferIds=hiddenOfferIds as |navOffers hasPrime showAllOffers hiddenOfferIds claimedOfferIds|}}
      {{#each navOffers as |offer|}}
        {{twitch-prime/offer-item offer=offer hiddenOfferIds=hiddenOfferIds}}
      {{/each}}
    {{/twitch-prime/offer-list}}
  `);

  return wait().then(() => {
    assert.elementCount(this.$('.offer-item'), expectedCount);
  });
});


test('coming from TYP', function(assert) {
  this.set('urlParams', { expandoffers: true });

  assert.expect(2);

  let offersCount = 4;
  this.server.createList('offer', offersCount);


  this.render(hbs`
    {{#twitch-prime/offer-list urlParams=urlParams as |navOffers hasPrime showAllOffers|}}
      {{#each navOffers as |offer|}}
        {{twitch-prime/offer-item offer=offer showAllOffers=showAllOffers}}
      {{/each}}
    {{/twitch-prime/offer-list}}
  `);

  return wait().then(() => {
    assert.elementCount(this.$('.offer-list__pr-nav-header'), 1, 'Expanded Left Nav Title');
    assert.elementCount(this.$('.warp__item--thumbnail'), offersCount, 'All offers are shown');
  });
});
