import { test } from 'ember-qunit';
import moduleForIntegration from 'web-client/tests/helpers/module-for-integration';
import { copy } from 'ember-metal/utils';
import { assign } from 'ember-platform';

import { POSSIBLE_BADGES } from 'web-client/controllers/dashboards/subscription-dash/badges';

let model, controller;

moduleForIntegration('controller:dashboards/subscription-dash/badges', 'Integration | Controller | dashboards/subscription-dash/badges', {
  beforeEach() {
    model = [
      {
        id: 12345,
        required_tenure_months: 6,
        image_1x_url: '#',
        image_2x_url: '#',
        image_4x_url: '#',
        title: '6-Month Subscriber'
      },
      {
        id: 54321,
        required_tenure_months: 24,
        image_1x_url: '#',
        image_2x_url: '#',
        image_4x_url: '#',
        title: '2-Year Subscriber'
      }
    ];
  }
});

test('maps a list of possible badges with passed-in, existing badges', function(assert) {
  assert.expect(POSSIBLE_BADGES.length);

  controller = this.subject({ model: model });

  let badgeList = controller.get('badgeList');
  let existingBadgeMap = {};
  let possibleBadge, possibleBadgeClone, existingBadge, expectedBadge;

  model.forEach(badge => {
    existingBadgeMap[badge.required_tenure_months] = badge;
  });

  for (let i = 0; i < POSSIBLE_BADGES.length; i++) {
    possibleBadge = POSSIBLE_BADGES[i];
    possibleBadgeClone = copy(possibleBadge);
    existingBadge = existingBadgeMap[possibleBadge.required_tenure_months];
    expectedBadge = assign(possibleBadgeClone, existingBadge);

    assert.propEqual(badgeList[i], expectedBadge, `"${expectedBadge.name}" badge is mapped`);
  }
});

test('default selected badge has the default subscriber badge data', function (assert) {
  assert.expect(3);

  let newBadge = {
    id: 12345,
    required_tenure_months: 0,
    image_1x_url: '#',
    image_2x_url: '#',
    image_4x_url: '#',
    title: 'Subscriber'
  };

  controller = this.subject({ model: [] });

  assert.propEqual(controller.get('selectedBadge'), POSSIBLE_BADGES[0], 'selected badge is the first in the list');

  controller.set('model', model);

  assert.propEqual(controller.get('selectedBadge'), POSSIBLE_BADGES[0], 'selected badge is still the first in the list');

  controller.get('model').addObject(newBadge);

  let expectedBadge = assign(newBadge, controller.get('selectedBadge'));

  assert.propEqual(controller.get('selectedBadge'), expectedBadge, 'selected badge is updated');
});

test('selects the correct badge by index', function (assert) {
  assert.expect(2);

  let existingBadge = model[0];
  let existingBadgeIndex;

  for (let i = 0; i < POSSIBLE_BADGES.length; i++) {
    if (POSSIBLE_BADGES[i].required_tenure_months === existingBadge.required_tenure_months) {
      existingBadgeIndex = i;
      break;
    }
  }

  controller = this.subject({ model: model });

  assert.propEqual(controller.get('selectedBadge'), POSSIBLE_BADGES[0], 'selected badge is the first in the list');

  controller.set('selectedIndex', existingBadgeIndex);

  let expectedBadge = assign(existingBadge, controller.get('selectedBadge'));

  assert.propEqual(controller.get('selectedBadge'), expectedBadge, 'selected badge is updated');
});

test('updates the list of badges when a new badge is added', function (assert) {
  assert.expect(2);

  let createdBadge = {
    id: 3,
    required_tenure_months: 12,
    image_1x_url: '#',
    image_2x_url: '#',
    image_4x_url: '#',
    title: '1-Year Subscriber'
  };

  controller = this.subject({ model: model });

  let existingBadges = controller.get('badgeList').filter(badge => !!badge.id);

  assert.elementCount(existingBadges, 2, 'only two existing badges are found in the list');

  controller.send('onBadgeCreate', createdBadge);

  existingBadges = controller.get('badgeList').filter(badge => !!badge.id);

  assert.elementCount(existingBadges, 3, 'three existing badges are found in the list');
});

test('updates the list of badges when an existing badge is removed', function (assert) {
  assert.expect(2);

  let deletedBadge = {
    id: 12345,
    required_tenure_months: 6
  };

  controller = this.subject({ model: model });

  let existingBadges = controller.get('badgeList').filter(badge => !!badge.id);

  assert.elementCount(existingBadges, 2, 'two existing badges are found in the list');

  controller.send('onBadgeDelete', deletedBadge);

  existingBadges = controller.get('badgeList').filter(badge => !!badge.id);

  assert.elementCount(existingBadges, 1, 'one existing badge is found in the list');
});
