import Service from 'ember-service';
import { moduleForModel, test } from 'ember-qunit';
import {
  setup as setupMirage, teardown as teardownMirage
} from 'web-client/tests/helpers/setup-mirage-for-integration';
import run from 'ember-runloop';
import { stubLogin } from 'web-client/tests/helpers/stub-login';
import { USER_TYPE_USER, USER_TYPE_STAFF } from 'web-client/utilities/user';

moduleForModel('user', 'Integration | Models | User', {
  integration: true,
  beforeEach() {
    setupMirage(this);
  },

  afterEach() {
    teardownMirage(this);
  }
});

test('fetching user data for the current session user', function(assert) {
  assert.expect(11);

  this.user = this.server.create('user', { type: USER_TYPE_STAFF, bio: 'wow!' });
  this.register('service:login', Service.extend({
    userData: { login: this.user.login }
  }));

  this.server.get('https://api.twitch.tv/kraken/user', () => {
    assert.ok(true, 'signed-in-user URL called');
    return this.user.attrs;
  });

  return run(() => {
    return this.store().findRecord('user', this.user.login);
  }).then(user => {
    assert.equal(user.get('id'), this.user.login, 'serializes id');
    assert.equal(user.get('bio'), this.user.bio, 'serializes bio');
    assert.equal(user.get('displayName'), this.user.display_name, 'serializes display name');
    assert.equal(user.get('logo'), this.user.logo, 'serializes logo');
    assert.equal(user.get('name'), this.user.name, 'serializes name');
    assert.strictEqual(user.get('staff'), true, 'serializes staff');

    // current-user only
    assert.equal(user.get('email'), this.user.email, 'serializes email for current user');
    assert.deepEqual(user.get('notifications'), { push: false, email: true }, 'serializes notifications for current user');
    assert.strictEqual(user.get('partnered'), true, 'serializes partnered for current user');
    assert.equal(user.get('type'), USER_TYPE_STAFF, 'serializes type');
  });
});

test("fetching data when the user isn't this session's", function(assert) {
  assert.expect(7);

  this.user = this.server.create('user', { type: USER_TYPE_USER });
  this.server.get('https://api.twitch.tv/kraken/users/twitch', () => {
    assert.ok(true, 'users URL called');
    return this.user.attrs;
  });

  return run(() => {
    return this.store().findRecord('user', this.user.login);
  }).then((user) => {
    assert.equal(user.get('id'), this.user.login, 'serializes id');
    assert.equal(user.get('bio'), this.user.bio, 'serializes bio');
    assert.equal(user.get('displayName'), this.user.display_name, 'serializes display name');
    assert.equal(user.get('logo'), this.user.logo, 'serializes logo');
    assert.equal(user.get('name'), this.user.name, 'serializes name');
    assert.equal(user.get('staff'), false, 'serializes staff');
  });
});

test("updating the session user's data", function(assert) {
  assert.expect(2);

  this.user = this.server.create('user', { type: USER_TYPE_STAFF });
  stubLogin();

  let newBio = 'this has changed';

  return run(() => {
    return this.store().findRecord('user', this.user.login);
  }).then((user) => {
    run(() => {
      user.set('bio', newBio);
    });
    return user.save();
  }).then((user) => {
    assert.equal(user.get('bio'), newBio, 'bio updated on user model');
    assert.equal(this.user.reload().bio, newBio, 'bio persisted to DB');
  });
});
