import { skip } from 'ember-qunit';
import { test } from 'ember-qunit';
import run from 'ember-runloop';
import RSVP from 'rsvp';
import moduleForModel from 'web-client/tests/helpers/module-for-model';
import {
  setup as setupMirage, teardown as teardownMirage
} from 'web-client/tests/helpers/setup-mirage-for-integration';
import playlistScenario from 'web-client/mirage/scenarios/video-playlist';
import { STATE_PUBLIC, STATE_PRIVATE } from 'web-client/utilities/video/playlist';
import { parseQueryParams } from 'web-client/utilities/url-params';
import { JTV_USER_PICTURES_404_USER_150X150_URL } from 'web-client/utilities/urls/static-cdn';

moduleForModel('video-playlist', 'Integration | Models | video-playlist', {
  integration: true,
  beforeEach() {
    setupMirage(this);
  },
  afterEach() {
    teardownMirage(this);
  }
});

skip('finding a playlist record', function(assert) {
  assert.expect(5);

  let savedPlaylist = playlistScenario(this.server);

  return run(() => {
    return this.store().findRecord('video-playlist', savedPlaylist.id);
  }).then((playlist) => {
    assert.equal(playlist.id, savedPlaylist.id, 'should properly serialize the ID');
    assert.equal(playlist.get('state'), STATE_PUBLIC, 'should properly serialize a simple attr');
    assert.equal(playlist.get('updatedAt'), savedPlaylist.updated_at, 'should properly serialize a snake_case attr');

    return RSVP.hash({
      // owner: playlist.get('owner'),
      items: playlist.get('items')
    });
  // }).then(({ owner, items }) => {
  }).then(({ items }) => {
    // assert.equal(owner.id, savedPlaylist.owner_id, 'should set the correct owner relationship id');
    // assert.equal(owner.get('displayName'), 'Twitch', 'should be able to retrieve owner info');

    assert.equal(items.length, savedPlaylist.items.length, 'should properly return items');
    items.forEach((item, index) => {
      assert.equal(item.get('position'), index, 'items should be ordered correctly');
    });
  });
});

test('querying for playlists by user ID when they have playlists', function(assert) {
  assert.expect(1);

  let ownerId = '12345';
  playlistScenario(this.server, { owner_id: ownerId });
  playlistScenario(this.server, { owner_id: ownerId });
  playlistScenario(this.server, { owner_id: ownerId });

  return run(() => {
    return this.store().query('video-playlist', { userId: ownerId });
  }).then((playlists) => {
    assert.equal(playlists.content.length, 3, 'should succesfully normalize both playlists');
  });
});

test('querying for playlists by user ID when they have no playlists', function(assert) {
  assert.expect(1);

  return run(() => {
    return this.store().query('video-playlist', { userId: '12345' });
  }).then((playlists) => {
    assert.equal(playlists.content.length, 0, 'should succesfully normalize both playlists');
  });
});

skip('saving a new playlist', function(assert) {
  assert.expect(8);

  let thumbnailUrl = JTV_USER_PICTURES_404_USER_150X150_URL;
  let expectedServerParams = {
    owner_id: '9999',
    state: STATE_PRIVATE,
    title: 'My New Playlist',
    thumbnail_url: thumbnailUrl,
    updated_at: '',
    views: '0'
  };

  let updatedAt = (new Date()).getTime();

  this.server.post('http://api.twitch.tv/kraken/playlists', function(schema, request) {
    assert.ok(true, 'POST was received successfully');

    let params = parseQueryParams(request.requestBody);
    delete params.on_site;
    assert.deepEqual(params, expectedServerParams, 'should send the expected payload to the server');

    params.id = '98765';
    params.items = [];
    params.updated_at = updatedAt;
    return params;
  }, 201);

  return run(() => {
    let playlist = this.store().createRecord('video-playlist', { title: expectedServerParams.title });
    let owner = this.store().createRecord('user', { id: expectedServerParams.owner_id });

    playlist.set('owner', owner);
    return playlist.save();
  }).then((savedPlaylist) => {
    assert.ok(true, 'saving should resolve succesfully');

    assert.equal(savedPlaylist.get('title'), expectedServerParams.title, 'should have a set title');
    assert.equal(savedPlaylist.get('updatedAt'), updatedAt, 'should set updateAt');
    assert.equal(savedPlaylist.get('views'), 0, 'views should default to 0');
    assert.equal(savedPlaylist.get('state'), STATE_PRIVATE, 'should have private state by default');
    assert.equal(savedPlaylist.get('thumbnailUrl'), thumbnailUrl, 'should have the proper thumbnail url');
  });
});

skip('updating an existing playlist', function(assert) {
  assert.expect(5);

  let ownerId = '9999';
  let playlist = playlistScenario(this.server, { owner_id: ownerId });
  let expectedTitle = 'fun new title';

  this.server.put('http://api.twitch.tv/kraken/playlists/:id', function(schema, request) {
    let playlistId = request.params.id;

    assert.ok(true, 'PUT was received successfully');
    assert.equal(playlistId, playlist.id, 'should update the correct ID');

    let params = parseQueryParams(request.requestBody);
    assert.equal(params.title, expectedTitle, 'should send the expected title to the server');

    params.id = playlistId;
    return params;
  }, 201);

  return run(() => {
    return this.store().findRecord('video-playlist', playlist.id);
  }).then((fetchedPlaylist) => {
    fetchedPlaylist.set('title', expectedTitle);
    return fetchedPlaylist.save();
  }).then((updatedPlaylist) => {
    assert.ok(true, 'should save succesfully');
    assert.equal(updatedPlaylist.get('title'), expectedTitle, 'should have the expected title');
  });
});
