import moduleForComponent from 'web-client/tests/helpers/module-for-component';
import { test } from 'ember-qunit';
import { DATERANGE_OPTIONS, MAX_DAYS_AGO } from 'web-client/components/dashboards/video-stats/date-range-picker';
import sinon from 'sinon';
import moment from 'moment';

const TODAY = new Date(1991, 3, 7);

moduleForComponent('dashboards/video-stats/date-range-picker', 'Unit | Component | dashboards/video-stats/date-range-picker', {
  unit: true,
  beforeEach() {
    this.setStartDateAction = sinon.spy();
    this.setEndDateAction = sinon.spy();

    this.component = this.subject({
      today: TODAY,
      startDate: new Date(),
      endDate: new Date(),
      setStartDate: this.setStartDateAction,
      setEndDate: this.setEndDateAction
    });
  }
});

test('default state values', function(assert) {
  this.render();

  assert.equal(
    this.component.get('selectedDateRange'), DATERANGE_OPTIONS[0],
    'should select the first date range option'
  );
  assert.equal(
    this.setStartDateAction.args[0][0], new Date(1991, 3, 5).toString(),
    'should dispatch a setStartDate action'
  );
  assert.equal(
    this.setEndDateAction.args[0][0], TODAY.toString(),
    'should dispatch a setEndDateAction'
  );
  assert.equal(
    this.component.get('isCustomRange'), false,
    'the default range should not be custom'
  );
});

test('computing the minimumDate for the date range pickers', function(assert) {
  assert.equal(
    this.component.get('minimumDate'), moment(TODAY).subtract(MAX_DAYS_AGO, 'days').toDate().toString(),
    'should be exactly a year before the today value'
  );
});

test('when the setStartDate action value is already selected', function(assert) {
  let initialSelectedRange = this.component.get('selectedDateRange');
  this.component.send('setStartDate', this.component.get('startDate'));

  let selectedDateRange = this.component.get('selectedDateRange');
  assert.equal(selectedDateRange, initialSelectedRange, 'should not have changed the current selected date range');
  assert.notOk(this.component.get('isCustomRange'), 'should not mark the current range as custom');
});

test('when the setStartDate action value is new', function(assert) {
  let expectedStartDate = new Date(1991, 2, 24);
  this.component.send('setStartDate', expectedStartDate);

  assert.deepEqual(this.component.get('selectedDateRange'), {}, 'should have an empty selectedDateRange');
  assert.equal(this.component.get('isCustomRange'), true, 'should set the custom date range property to true');
  assert.equal(this.setStartDateAction.args[0][0], expectedStartDate, 'should send the setStartDate action');
});

test('when the setEndDate action value is already selected', function(assert) {
  let initialSelectedRange = this.component.get('selectedDateRange');
  this.component.send('setEndDate', this.component.get('endDate'));

  assert.equal(this.component.get('selectedDateRange'), initialSelectedRange, 'should not change selectedDateRange');
  assert.notOk(this.component.get('isCustomRange'), 'should not mark the current range as custom');
});

test('when the setEndDate action value is new', function(assert) {
  let expectedEndDate = new Date(1991, 2, 29);
  this.component.send('setEndDate', expectedEndDate);

  assert.equal(this.component.get('isCustomRange'), true, 'should set the custom date range property to true');
  assert.deepEqual(this.component.get('selectedDateRange'), {}, 'should have an empty selectedDateRange');
  assert.equal(this.setEndDateAction.args[0][0], expectedEndDate, 'should send the setEndDate action');
});

test('setDateRange helper function', function(assert) {
  let expectedDateRange = DATERANGE_OPTIONS[0];
  this.component.setDateRange(expectedDateRange);

  assert.equal(this.component.get('selectedDateRange'), expectedDateRange, 'should properly set a new data range');
  assert.equal(this.component.get('isCustomRange'), false, 'should set the custom date range property to true');
});

test('rendering the component', function(assert) {
  let selectedRangeValue = this.$('select').val();

  let expectedRange = this.component.get('selectedDateRange');
  assert.equal(selectedRangeValue, expectedRange.value, 'the select element should mark the correct option as selected');
});

test('when a new date range is selected via input', function(assert) {
  this.$('.js-stats-range-picker').val('thisWeek');
  this.$('.js-stats-range-picker').change();

  let selectedDateRange = this.component.get('selectedDateRange');
  assert.equal(selectedDateRange.value, 'thisWeek', 'should update the current selected date range');
});

test('when a custom start date is selected via pikaday', function(assert) {
  this.component.send('setStartDate', new Date(1991, 3, 4));

  assert.equal(this.$('.js-stats-range-picker').val(), null, 'should have an empty selector value');
  assert.equal(this.$('.js-stats-range-picker option:selected').text(), 'Custom Range', 'should have a custom range placeholder');
});

moduleForComponent('dashboards/video-stats/date-range-picker', 'Unit | Component | dashboards/video-stats/date-range-picker', {
  unit: true
});

test('minimumDate is respected when passed in', function(assert) {
  let minDate = new Date(1991, 3, 1);

  let myComponent = this.subject({
    today: TODAY,
    startDate: new Date(),
    endDate: new Date(),
    setStartDate: this.setStartDateAction,
    setEndDate: this.setEndDateAction,
    minimumDate: minDate
  });

  assert.equal(myComponent.get('minimumDate'), moment(minDate).toDate().toString());
});

test('minimumDate does not go back more than 90 days', function(assert) {
  let minDate = new Date(1990, 3, 7);

  let myComponent = this.subject({
    today: TODAY,
    startDate: new Date(),
    endDate: new Date(),
    setStartDate: this.setStartDateAction,
    setEndDate: this.setEndDateAction,
    minimumDate: minDate
  });

  assert.equal(myComponent.get('minimumDate'), moment(TODAY).subtract(MAX_DAYS_AGO, 'days').toDate().toString());
});
