import moduleFor from 'web-client/tests/helpers/module-for';
import { test } from 'ember-qunit';
import RSVP from 'rsvp';
import Service from 'ember-service';
import { AuthenticatedSession } from 'web-client/tests/helpers/mock-session-service';

moduleFor('service:bits-room', 'Unit | Service | bits room', {
  beforeEach() {
    this.owner.register('service:session', AuthenticatedSession);
  }
});

test('Calculate Bits Recipient Eligibility where recipient is eligible', function (assert) {
  assert.expect(8);

  let eligibility = {
    eligible: true,
    minBits: 5
  };

  let recipientEligibility = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'loadRecipientEligibility was called');
      resolve(eligibility);
    });
  };
  let adsEnabled = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'ads enabled check was called');
      resolve(true);
    });
  };

  this.owner.register('service:bits', Service.extend({
    loadRecipientEligibility: recipientEligibility
  }));
  this.owner.register('service:bits-ads-enabled', Service.extend({
    isEnabled: adsEnabled
  }));
  let bitsRoom = this.subject();

  let seenPromoKey = false;
  let seenAdsForBitsPromoKey = true;
  let seenRecipientCanEnablePromoKey = false;

  return bitsRoom._calcRecipientEligibility({
    seenPromoKey,
    seenAdsForBitsPromoKey,
    seenRecipientCanEnablePromoKey
  }).then(function() {
    assert.equal(bitsRoom.get('isRecipientIneligible'), false, 'recipient should be eligible');
    assert.ok(bitsRoom.get('needsToSeeFTUE'), 'bits promo should be shown');
    assert.equal(bitsRoom.get('needsToSeeAdsFTUE'), false, 'bits for ads promo should not be shown');
    assert.equal(bitsRoom.get('needsToSeeRecipientCanEnableFTUE'), false, 'recipient can enable bits ftue should not be shown');
    assert.ok(bitsRoom.get('isHelperShown'), 'bits helper should be shown');
    assert.equal(bitsRoom.get('minimumBits'), 5, 'minimum bits should be set');
  });
});

test('Calculate Bits Recipient Eligibility where recipient is eligible and ads for bits promo enabled', function (assert) {
  assert.expect(9);

  let eligibility = {
    eligible: true,
    minBits: 5
  };

  let recipientEligibility = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'loadRecipientEligibility was called');
      resolve(eligibility);
    });
  };
  let adsEnabled = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'ads enabled check was called');
      resolve(true);
    });
  };
  let bitsAdsAvailable = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'bitsAdsAvailable was called');
      resolve({});
    });
  };

  this.owner.register('service:bits', Service.extend({
    loadRecipientEligibility: recipientEligibility,
    bitsAdsAvailable: bitsAdsAvailable
  }));
  this.owner.register('service:bits-ads-enabled', Service.extend({
    isEnabled: adsEnabled
  }));
  let bitsRoom = this.subject();

  let seenPromoKey = true;
  let seenAdsForBitsPromoKey = false;
  let seenRecipientCanEnablePromoKey = false;

  bitsRoom.set('isPromoEnabled', true);

  return bitsRoom._calcRecipientEligibility({
    seenPromoKey,
    seenAdsForBitsPromoKey,
    seenRecipientCanEnablePromoKey
  }).then(() => {
    assert.equal(bitsRoom.get('isRecipientIneligible'), false, 'recipient should be eligible');
    assert.equal(bitsRoom.get('needsToSeeFTUE'), false, 'bits promo should not be shown');
    assert.ok(bitsRoom.get('needsToSeeAdsFTUE'), 'ads for bits promo should be shown');
    assert.equal(bitsRoom.get('needsToSeeRecipientCanEnableFTUE'), false, 'recipient can enable bits ftue should not be shown');
    assert.ok(bitsRoom.get('isHelperShown'), 'bits helper should be shown');
    assert.equal(bitsRoom.get('minimumBits'), 5, 'minimum bits should be set');
  });
});

test('Calculate Bits Recipient Eligibility where recipient is not eligible', function (assert) {
  assert.expect(8);

  let eligibility = {
    eligible: false,
    minBits: 5
  };

  let recipientEligibility = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'loadRecipientEligibility was called');
      resolve(eligibility);
    });
  };
  let adsEnabled = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'ads enabled check was called');
      resolve(true);
    });
  };

  this.owner.register('service:bits', Service.extend({
    loadRecipientEligibility: recipientEligibility
  }));
  this.owner.register('service:bits-ads-enabled', Service.extend({
    isEnabled: adsEnabled
  }));
  let bitsRoom = this.subject();

  let seenPromoKey = true;
  let seenAdsForBitsPromoKey = true;
  let seenRecipientCanEnablePromoKey = false;

  return bitsRoom._calcRecipientEligibility({
    seenPromoKey,
    seenAdsForBitsPromoKey,
    seenRecipientCanEnablePromoKey
  }).then(function() {
    assert.ok(bitsRoom.get('isRecipientIneligible'), 'recipient should not be eligible');
    assert.equal(bitsRoom.get('needsToSeeFTUE'), false, 'bits promo should not be shown');
    assert.equal(bitsRoom.get('needsToSeeAdsFTUE'), false, 'ads for bits promo should not be shown');
    assert.equal(bitsRoom.get('needsToSeeRecipientCanEnableFTUE'), false, 'recipient can enable bits ftue should not be shown');
    assert.equal(bitsRoom.get('isHelperShown'), false, 'bits helper should not be shown');
    assert.equal(bitsRoom.get('minimumBits'), 5, 'minimum bits should be set');
  });
});

test('Calculate Bits Recipient Eligibility loadRecipientEligibility exception', function (assert) {
  assert.expect(8);

  let recipientEligibility = () => {
    return new RSVP.Promise((resolve, reject) => {
      assert.ok(true, 'loadRecipientEligibility was called');
      reject();
    });
  };
  let adsEnabled = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'ads enabled check was called');
      resolve(true);
    });
  };

  this.owner.register('service:bits', Service.extend({
    loadRecipientEligibility: recipientEligibility
  }));
  this.owner.register('service:bits-ads-enabled', Service.extend({
    isEnabled: adsEnabled
  }));
  let bitsRoom = this.subject();

  let seenPromoKey = false;
  let seenAdsForBitsPromoKey = false;
  let seenRecipientCanEnablePromoKey = false;

  return bitsRoom._calcRecipientEligibility({
    seenPromoKey,
    seenAdsForBitsPromoKey,
    seenRecipientCanEnablePromoKey
  }).then(function() {
    assert.ok(bitsRoom.get('isRecipientIneligible'), 'recipient should be ineligible');
    assert.equal(bitsRoom.get('needsToSeeFTUE'), false, 'bits promo should not be shown');
    assert.equal(bitsRoom.get('needsToSeeAdsFTUE'), false, 'ads for bits promo should not be shown');
    assert.equal(bitsRoom.get('needsToSeeRecipientCanEnableFTUE'), false, 'recipient can enable bits ftue should not be shown');
    assert.equal(bitsRoom.get('isHelperShown'), false, 'bits helper should not be shown');
    assert.equal(bitsRoom.get('minimumBits'), 1, 'minimum bits should be 1');
  });
});

test('Calculate Bits Recipient Eligibility recipient ineligible but can enable', function (assert) {
  assert.expect(7);

  let channelName = 'partner';

  let recipientEligibility = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'loadRecipientEligibility was called');
      resolve({
        eligible: false
      });
    });
  };
  let adsEnabled = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'ads enabled check was called');
      resolve(false);
    });
  };

  let sessionSvc = this.owner.lookup('service:session');
  sessionSvc.set('userData', {
    login: channelName,
    is_partner: true
  });
  sessionSvc.set('userModel.id', channelName);

  this.owner.register('service:bits', Service.extend({
    loadRecipientEligibility: recipientEligibility
  }));

  this.owner.register('service:bits-ads-enabled', Service.extend({
    isEnabled: adsEnabled
  }));
  let bitsRoom = this.subject({
    channelName: channelName
  });

  let seenPromoKey = false;
  let seenAdsForBitsPromoKey = false;
  let seenRecipientCanEnablePromoKey = false;

  return bitsRoom._calcRecipientEligibility({
    seenPromoKey,
    seenAdsForBitsPromoKey,
    seenRecipientCanEnablePromoKey
  }).then(() => {
    assert.ok(bitsRoom.get('isRecipientIneligible'), 'recipient should be ineligible');
    assert.equal(bitsRoom.get('needsToSeeFTUE'), false, 'bits promo should not be shown');
    assert.equal(bitsRoom.get('needsToSeeAdsFTUE'), false, 'ads for bits promo should not be shown');
    assert.ok(bitsRoom.get('needsToSeeRecipientCanEnableFTUE'), 'recipient can enable bits ftue should be shown');
    assert.ok(bitsRoom.get('isHelperShown'), 'bits helper should be shown');
  });
});

test('Calculate Bits Recipient Eligibility isRecipientBitsIneligible already true', function (assert) {
  let bitsRoom = this.subject();
  let seenPromoKey = false;
  let seenAdsForBitsPromoKey = false;
  let seenRecipientCanEnablePromoKey = false;

  bitsRoom.set('isRecipientIneligible', true);
  bitsRoom._calcRecipientEligibility({
    seenPromoKey,
    seenAdsForBitsPromoKey,
    seenRecipientCanEnablePromoKey
  });

  assert.equal(bitsRoom.get('needsToSeeFTUE'), false, 'bits promo should not be shown');
  assert.equal(bitsRoom.get('needsToSeeAdsFTUE'), false, 'ads for bits promo should not be shown');
  assert.equal(bitsRoom.get('needsToSeeRecipientCanEnableFTUE'), false, 'recipient can enable bits ftue should not be shown');
  assert.equal(bitsRoom.get('isHelperShown'), false, 'bits helper should not be shown');
  assert.equal(bitsRoom.get('minimumBits'), 1, 'minimum bits should be 1');
});

test('Calculate Bits Recipient Eligibility where recipient is eligible and ads are not available', function (assert) {
  assert.expect(8);

  let eligibility = {
    eligible: true,
    minBits: 5
  };

  let recipientEligibility = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'loadRecipientEligibility was called');
      resolve(eligibility);
    });
  };
  let adsEnabled = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'ads enabled check was called');
      resolve(true);
    });
  };
  let bitsAdsAvailable = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'bitsAdsAvailable was called');
      resolve({ error: "error" });
    });
  };

  this.owner.register('service:bits', Service.extend({
    loadRecipientEligibility: recipientEligibility,
    bitsAdsAvailable: bitsAdsAvailable
  }));
  this.owner.register('service:bits-ads-enabled', Service.extend({
    isEnabled: adsEnabled
  }));
  let bitsRoom = this.subject();

  let seenPromoKey = true;
  let seenAdsForBitsPromoKey = false;
  let seenRecipientCanEnablePromoKey = false;

  bitsRoom.set('isPromoEnabled', true);

  return bitsRoom._calcRecipientEligibility({
    seenPromoKey,
    seenAdsForBitsPromoKey,
    seenRecipientCanEnablePromoKey
  }).then(() => {
    assert.equal(bitsRoom.get('isRecipientIneligible'), false, 'recipient should be eligible');
    assert.equal(bitsRoom.get('needsToSeeFTUE'), false, 'bits promo should not be shown');
    assert.equal(bitsRoom.get('needsToSeeAdsFTUE'), false, 'ads for bits promo should not be shown');
    assert.ok(bitsRoom.get('isHelperShown'), 'bits helper should be shown');
    assert.equal(bitsRoom.get('minimumBits'), 5, 'minimum bits should be set');
  });
});

test('Calculate Bits Recipient Eligibility where hashtags are set', function (assert) {
  assert.expect(8);

  let hashtagsData = [{"hashtag":"daigo","selected":true,"total":1}];

  let eligibility = {
    eligible: true,
    hashtags: hashtagsData
  };

  let recipientEligibility = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'loadRecipientEligibility was called');
      resolve(eligibility);
    });
  };
  let adsEnabled = () => {
    return new RSVP.Promise((resolve) => {
      assert.ok(true, 'ads enabled check was called');
      resolve(true);
    });
  };

  this.owner.register('service:bits', Service.extend({
    loadRecipientEligibility: recipientEligibility
  }));
  this.owner.register('service:bits-ads-enabled', Service.extend({
    isEnabled: adsEnabled
  }));
  let bitsRoom = this.subject();

  let seenPromoKey = false;
  let seenAdsForBitsPromoKey = true;
  let seenRecipientCanEnablePromoKey = false;

  return bitsRoom._calcRecipientEligibility({
    seenPromoKey,
    seenAdsForBitsPromoKey,
    seenRecipientCanEnablePromoKey
  }).then(function() {
    assert.equal(bitsRoom.get('isRecipientIneligible'), false, 'recipient should be eligible');
    assert.ok(bitsRoom.get('needsToSeeFTUE'), 'bits promo should be shown');
    assert.equal(bitsRoom.get('needsToSeeAdsFTUE'), false, 'bits for ads promo should not be shown');
    assert.equal(bitsRoom.get('needsToSeeRecipientCanEnableFTUE'), false, 'recipient can enable bits ftue should not be shown');
    assert.ok(bitsRoom.get('isHelperShown'), 'bits helper should be shown');
    assert.equal(bitsRoom.get('hashtags'), hashtagsData, 'minimum bits should be set');
  });
});

// TODO: add more tests
