import Service from 'ember-service';
import moduleFor from 'web-client/tests/helpers/module-for';
import { test } from 'ember-qunit';
import { AuthenticatedSession } from 'web-client/tests/helpers/mock-session-service';
import channelScenario from 'web-client/mirage/scenarios/channel';

moduleFor('service:bits', 'Unit | Service | bits', {
  beforeEach() {
    this.inject.service('api');
    this.inject.service('ajax');
    this.inject.service('crypto');
    this.owner.register('service:session', AuthenticatedSession);

    this.owner.register('service:bits-emotes', Service.extend({
      regexes: [
        {
          zeroValue: /^\s*cheer0+\s*$/i,
          valid: /^\s*cheer\d+\s*$/i
        },
        {
          zeroValue: /^\s*kappa0+\s*$/i,
          valid: /^\s*kappa\d+\s*$/i
        }
      ]
    }));
    this.channel = channelScenario(this.server, 'twitch-partner');

    let { bitsBalances, channels } = this.server.schema;
    let senderChannel = channels.first({channelName: AuthenticatedSession.name});
    let senderBitsBalance = bitsBalances.first({channelId: senderChannel.attrs.id});
    this.senderBalance = senderBitsBalance.attrs.balance;
  }
});

test('sendBits with cheer - successful', function(assert) {
  assert.expect(2);

  let cheerMessageAmount = this.senderBalance - 1;
  let service = this.subject();
  service.set('channelTotal', 0);

  return service.sendBits(this.channel._id, `cheer${cheerMessageAmount}`).then(() => {
    assert.equal(service.get('channelTotal'), cheerMessageAmount, 'Channel total was updated');
    assert.equal(service.get('balance'), 1, 'Balance was updated');
  });
});

test('sendBits with kappa - successful', function(assert) {
  assert.expect(2);

  let cheerMessageAmount = this.senderBalance - 1;
  let service = this.subject();
  service.set('channelTotal', 0);

  return service.sendBits(this.channel._id, `kappa${cheerMessageAmount}`).then(() => {
    assert.equal(service.get('channelTotal'), cheerMessageAmount, 'Channel total was updated');
    assert.equal(service.get('balance'), 1, 'Balance was updated');
  });
});

test('sendBits with cheer and kappa - successful', function(assert) {
  assert.expect(2);

  let service = this.subject();
  service.set('channelTotal', 0);

  return service.sendBits(this.channel._id, `cheer1 kappa1`).then(() => {
    assert.equal(service.get('channelTotal'), 2, 'Channel total was updated');
    assert.equal(service.get('balance'), this.senderBalance-2, 'Balance was updated');
  });
});

test('sendBits - error', function(assert) {
  this.server.post('https://api.twitch.tv/bits/events', {}, 401);
  assert.expect(2);

  let cheerMessageAmount = this.senderBalance - 1;
  let service = this.subject();
  service.set('channelTotal', 0);
  service.set('balance', this.senderBalance);

  return service.sendBits(this.channel._id, `cheer${cheerMessageAmount}`).catch(() => {
    assert.equal(service.get('channelTotal'), 0, `Channel total didn't change`);
    assert.equal(service.get('balance'), this.senderBalance, `Balance didn't change`);
  });
});

test('sendBits with incorrect regex - error', function(assert) {
  assert.expect(2);

  let cheerMessageAmount = this.senderBalance - 1;
  let service = this.subject();
  service.set('channelTotal', 0);

  return service.sendBits(this.channel._id, `foo${cheerMessageAmount}`).then(() => {
    assert.equal(service.get('channelTotal'), 0, `Channel total didn't change`);
    assert.equal(service.get('balance'), this.senderBalance, `Balance didn't change`);
  });
});
