import sinon from 'sinon';
import { moduleFor, test } from 'ember-qunit';
import EmberObject from 'ember-object';
import Service from 'ember-service';

moduleFor('service:persistent-player-content', 'Unit | Service | persistent-player-content',  {
  beforeEach() {
    this.mockPubsub = Service.extend({
      on: sinon.spy(),
      off: sinon.spy()
    });

    this.mockChannel = EmberObject.create({
      id: 'twitch',
      name: 'twitch'
    });

    this.mockVOD = EmberObject.create({
      id: '12345',
      channel: this.mockChannel
    });

    this.register('service:pubsub', this.mockPubsub);

    this.service = this.subject();
  }
});

test('when only currentChannel is set', function(assert) {
  this.service.set('currentChannel', this.mockChannel);
  this.service.set('currentVOD', null);

  assert.equal(this.service.get('targetChannel'), this.mockChannel, 'the target channel is the current channel');
  assert.ok(this.service.get('hasContent'), 'hasContent is true');

  this.mockChannel.set('playerIsLive', true);
  assert.ok(this.service.get('isLive'), 'isLive is true when the channel is live');

  this.mockChannel.set('playerIsLive', false);
  assert.notOk(this.service.get('isLive'), 'isLive is false when the channel is not live');
});

test('when the currentVOD is set', function(assert) {
  this.service.set('currentVOD', this.mockVOD);

  assert.equal(this.service.get('targetChannel'), this.mockVOD.get('channel'), 'the target channel is the vod owner`s channel');
  assert.ok(this.service.get('hasContent'), 'hasContent is true');
});

test('when the current channel is hosting', function(assert) {
  let hostedChannel = EmberObject.create({
    id: 'foilstormcrow'
  });

  this.mockChannel.set('hostModeTarget', hostedChannel);
  this.service.set('currentChannel', this.mockChannel);

  assert.equal(this.service.get('targetChannel'), hostedChannel, 'the target channel is the hosted channel');

  hostedChannel.set('playerIsLive', true);
  assert.ok(this.service.get('isLiveHost'), 'isLiveHost is true when the hosted channel is live');

  hostedChannel.set('playerIsLive', false);
  assert.notOk(this.service.get('isLiveHost'), 'isLiveHost is false when the hosted channel is not live');
});

test('when neither currentChannel nor currentVOD is set', function(assert) {
  assert.notOk(this.service.get('hasContent'), 'hasContent is false');
});

test('when currentChannel changes', function(assert) {
  this.service.set('currentChannel', this.mockChannel);

  assert.ok(this.service.get('pubsub.on').calledWith(
    'video-playback.twitch',
    this.service,
    this.service._handlePubsub
  ), 'it listens to the `video-playback` pubsub event');

  let secondChannel = EmberObject.create({
    id: 'foilstormcrow',
    name: 'foilstormcrow'
  });

  this.service.set('currentChannel', secondChannel);

  assert.ok(this.service.get('pubsub.on').calledWith(
    'video-playback.foilstormcrow',
    this.service,
    this.service._handlePubsub
  ), 'it listens to the `video-playback` pubsub event for the new channel');

  assert.ok(this.service.get('pubsub.off').calledWith('video-playback.twitch'), 'it unlistens to the `video-playback event for the old channel`');

});
