import EmberObject from 'ember-object';
import { isEmpty } from 'ember-utils';
import { test, moduleFor } from 'ember-qunit';
import { assign } from 'ember-platform';
import sinon from 'sinon';
import wait from 'ember-test-helpers/wait';

let service;

moduleFor('service:tracking', 'Unit | Service | tracking', {
  beforeEach() {
    service = this.subject();
  },
  afterEach() {
    sinon.restore();
  }
});

let firstStub, secondStub, firstService, event, props, defaultProps;

function setupTrackTests() {
  let services = service.get('services');
  firstService = services[0];

  firstStub = sinon.stub(firstService, 'trackEvent');
  secondStub = sinon.stub(services[1], 'trackEvent');

  event = 'foo';
  props = {};
  defaultProps = {platform: "web"};
}

test('trackEvent - all services', function (assert) {
  assert.expect(6);
  setupTrackTests.call(this);

  service.trackEvent({ event, data: props });

  return wait().then(() => {
    props = assign(defaultProps, props);
    assert.ok(firstStub.calledOnce, 'first service trackEvent was called');
    let { args: firstArgs } = firstStub.firstCall;
    assert.equal(firstArgs[0], event, 'event passed to first service');
    assert.deepEqual(firstArgs[1], props, 'data passed to first service');

    assert.ok(secondStub.calledOnce, 'second service trackEvent was called');
    let { args: secondArgs } = secondStub.firstCall;
    assert.equal(secondArgs[0], event, 'event passed to second service');
    assert.deepEqual(secondArgs[1], props, 'data passed to second service');
  });
});

test('trackEvent - some services', function (assert) {
  assert.expect(2);
  setupTrackTests.call(this);

  service.trackEvent({ services: [firstService.name], event, data: props });

  return wait().then(() => {
    assert.ok(firstStub.calledOnce, 'first service trackEvent was called');
    assert.notOk(secondStub.calledOnce, 'second service trackEvent was not called');
  });
});

let trackStub, id, followersTotal, channel;

function setupChannelTests(src) {
  trackStub = sinon.stub(service, 'trackEvent');
  event = 'foo';
  id = 'twitch';
  followersTotal = 4321;
  channel = EmberObject.create({
    id,
    followersTotal
  });
  props = {
    src,
    bar: 'baz'
  };
}

test('trackChannelEvent', function (assert) {
  assert.expect(5);
  setupChannelTests.call(this);

  service.trackChannelEvent(event, channel, props);

  return wait().then(() => {
    assert.ok(trackStub.calledOnce, 'trackEvent was called');

    let { args } = trackStub.firstCall;
    assert.equal(args[0].event, event, 'event passed to trackEvent');
    let { data } = args[0];
    assert.equal(data.bar, props.bar, 'data passed to trackEvent');
    assert.equal(data.follow_count, followersTotal, 'followersTotal added to data');
    assert.ok(isEmpty(data.host_channel), 'host_channel not added to data');
  });
});

test('trackChannelEvent host_mode', function (assert) {
  assert.expect(4);
  setupChannelTests.call(this, 'host_mode');

  service.trackChannelEvent(event, channel, props);

  return wait().then(() => {
    assert.ok(trackStub.calledOnce, 'trackEvent was called');

    let { args } = trackStub.firstCall;
    assert.equal(args[0].event, event, 'event passed to trackEvent');
    let { data } = args[0];
    assert.equal(data.bar, props.bar, 'data passed to trackEvent');
    assert.equal(data.follow_count, followersTotal, 'followersTotal added to data');
  });
});
