import { module, test } from 'qunit';
import validateEvent, { VALIDATION_MESSAGES, FILE_SIZE_LIMIT } from 'web-client/utilities/events/validate-event';

module('utilities/events/validate-event');

test('Valid events do not have errors', function(assert) {
  let validation = validateEvent({
    startTime: date({ hours: 1 }),
    endTime: date({ hours: 2 })
  }, {
    now: date({ hours: 0 })
  });

  assert.equal(validation.hasErrors, false, 'has no errors');
  assert.equal(validation.errors.startTime, undefined, 'no startTime error');
  assert.equal(validation.errors.endTime, undefined, 'no endTime error');
});

test('Events with no values has appropriate errors', function(assert) {
  let validation = validateEvent({});

  assert.equal(validation.hasErrors, true, 'has errors');
  assert.equal(validation.errors.startTime, VALIDATION_MESSAGES.startTimeMissing);
  assert.equal(validation.errors.endTime, VALIDATION_MESSAGES.endTimeMissing);
});

test('Warns when an event is starting in the past', function(assert) {
  let validation = validateEvent({
    startTime: date({ hours: 1 }),
    endTime: date({ hours: 3 })
  }, {
    now: date({ hours: 2 })
  });

  assert.equal(
    validation.warnings.startTime,
    VALIDATION_MESSAGES.eventStartsInPast,
    'sets warning on startTime'
  );
});

test('Warns when an event is starting in less than an hour from now', function(assert) {
  let validation = validateEvent({
    startTime: date({ hours: 2, minutes: 30 }),
    endTime: date({ hours: 3 })
  }, {
    now: date({ hours: 2 })
  });

  assert.equal(
    validation.warnings.startTime,
    VALIDATION_MESSAGES.eventStartsSoon,
    'sets warning on startTime'
  );
});

test('An event must last at least 30 minutes', function(assert) {
  let validation = validateEvent({
    startTime: date({ hours: 1, minutes: 0 }),
    endTime: date({ hours: 1, minutes: 29 })
  }, {
    now: date()
  });

  assert.equal(validation.hasErrors, true, 'has errors');
  assert.equal(
    validation.errors.endTime,
    VALIDATION_MESSAGES.eventTooShort,
    'sets error on endTime'
  );
});

test('warns when an imageFile is the wrong type', function(assert) {
  let validation = validateEvent({
    imageFile: { type: 'image/gif' }
  });

  assert.equal(
    validation.warnings.imageFile,
    VALIDATION_MESSAGES.wrongFileType,
    'sets warning on image'
  );
});

test('warns when an imageFile is too big', function(assert) {
  let validation = validateEvent({
    imageFile: { type: 'image/png', size: FILE_SIZE_LIMIT + 1 }
  });

  assert.equal(
    validation.warnings.imageFile,
    VALIDATION_MESSAGES.imageTooBig,
    'sets warning on image'
  );
});

test('has error when has gameName but no gameId', function(assert) {
  let validation = validateEvent({
    gameName: 'Overwatch',
    gameId: null
  });

  assert.equal(
    validation.errors.game,
    VALIDATION_MESSAGES.gameInvalid,
    'sets error on game'
  );
});

test('no game error when with gameName and gameId', function(assert) {
  let validation = validateEvent({
    gameName: 'Overwatch',
    gameId: 1
  });

  assert.equal(validation.errors.game, undefined, 'has no game error');
});

test('no game error when missing gameName and gameId', function(assert) {
  let validation = validateEvent({
    gameName: '',
    gameId: null
  });

  assert.equal(validation.errors.game, undefined, 'has no game error');
});

function date(params = {}) {
  return new Date(
    params.year || 2017,
    params.month || 0,
    params.day || 1,
    params.hours || 0,
    params.minutes || 0,
    params.seconds || 0,
    params.microseconds || 0
  );
}
