import { module, test } from 'qunit';
import moment from 'moment';
import DetailsReport, { HOURLY_AGGREGATION_CUTOFF } from 'web-client/utilities/video-stats/details-report';
import { getTimeSeriesData } from 'web-client/mirage/fixtures/video-stats-timeseries';
import { AGGREGATION_HOURLY, AGGREGATION_DAILY } from 'web-client/models/video-stats/channel-timeseries';

const TODAY = new Date(1991, 3, 6);
const HOURLY_STARTDATE = moment(TODAY).subtract(HOURLY_AGGREGATION_CUTOFF - 1, 'hours');
const DAILY_STARTDATE = moment(TODAY).subtract(HOURLY_AGGREGATION_CUTOFF + 1, 'hours');

module('Unit | Utils | video-stats/details-report', {
  beforeEach() {
    this.subject = DetailsReport.create({
      today: TODAY,
      data: getTimeSeriesData()
    });
  }
});

test('default state', function(assert) {
  assert.ok(this.subject.get('selectedMetric'), 'should set a default selected metric');
  assert.notOk(this.subject.get('selectedFilter'), 'should not set a default selected filter');
});

test('when startDate is more than 48 hours ago', function(assert) {
  this.subject.set('startDate', DAILY_STARTDATE);
  assert.equal(this.subject.get('aggregationMode'), AGGREGATION_DAILY, 'should use daily aggregation');
});

test('when startDate is less than 48 hours ago', function(assert) {
  this.subject.set('startDate', HOURLY_STARTDATE);
  assert.equal(this.subject.get('aggregationMode'), AGGREGATION_HOURLY, 'should use daily aggregation');
});

test('timeSeries when the aggregation is hourly', function(assert) {
  this.subject.set('startDate', HOURLY_STARTDATE);
  let timeSeries = this.subject.get('timeSeries');
  let aggregations = this.subject.get('timeSeriesAggregations');

  assert.ok(aggregations, 'aggregations should not be empty');
  assert.equal(timeSeries, aggregations[AGGREGATION_HOURLY], 'data should be filtered correctly');
});

test('timeSeries when the aggregation is daily', function(assert) {
  this.subject.set('startDate', DAILY_STARTDATE);
  let timeSeries = this.subject.get('timeSeries');
  let aggregations = this.subject.get('timeSeriesAggregations');

  assert.ok(aggregations, 'aggregations should not be empty');
  assert.equal(timeSeries, aggregations[AGGREGATION_DAILY], 'data should be filtered correctly');
});

test('metricOptions', function(assert) {
  this.subject.set('startDate', DAILY_STARTDATE);
  let metricOptions = this.subject.get('metricOptions');

  assert.equal(metricOptions.length, 3, 'should return mw, avd, and viewcount options');
});

test('filterOptions', function(assert) {
  let filterOptions = this.subject.get('filterOptions');
  let filterValues = filterOptions.map(options => options.value);

  assert.deepEqual(filterValues, ['country', 'platform', 'referer'], 'should have proper filter options');
});

test('summaryTableTitle when no filter is selected', function(assert) {
  let summaryTableTitle = this.subject.get('summaryTableTitle');

  assert.notOk(summaryTableTitle, 'should be empty');
});

test('summaryTableTitle when a filter is selected', function(assert) {
  this.subject.set('selectedFilter', 'country');
  let summaryTableTitle = this.subject.get('summaryTableTitle');

  assert.equal(summaryTableTitle, 'Country', 'should append the filter to the header');
});

test('timeSeriesQuery when no filter is selected', function(assert) {
  assert.deepEqual(this.subject.get('timeSeriesQuery').length, 1, 'should have one row');
});

test('chartDataSets', function(assert) {
  this.subject.set('timeSeries', {
    query() {
      return [{
        key: 'some-key',
        items: [{
          value: 'some-value',
          timeStamp: new Date(2016, 11, 1, 13, 50, 50)
        }]
      }];
    }
  });

  assert.deepEqual(this.subject.get('chartDataSets'), {
    'some-key': [{
      x: 'Thu, 1PM',
      y: 'some-value'
    }]
  }, 'should convert the data sets into an object for chart.js');
});
