package cloudwatch

import (
	"os"
	"time"

	"code.justin.tv/vodsvc/aws/awsconfig"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	cwpkg "github.com/aws/aws-sdk-go/service/cloudwatch"
	log "go.uber.org/zap"
)

const (
	UNIT_Seconds          = "Seconds"
	UNIT_Microseconds     = "Microseconds"
	UNIT_Milliseconds     = "Milliseconds"
	UNIT_Bytes            = "Bytes"
	UNIT_Kilobytes        = "Kilobytes"
	UNIT_Megabytes        = "Megabytes"
	UNIT_Gigabytes        = "Gigabytes"
	UNIT_Terabytes        = "Terabytes"
	UNIT_Bits             = "Bits"
	UNIT_Kilobits         = "Kilobits"
	UNIT_Megabits         = "Megabits"
	UNIT_Gigabits         = "Gigabits"
	UNIT_Terabits         = "Terabits"
	UNIT_Percent          = "Percent"
	UNIT_Count            = "Count"
	UNIT_Bytes_Second     = "Bytes/Second"
	UNIT_Kilobytes_Second = "Kilobytes/Second"
	UNIT_Megabytes_Second = "Megabytes/Second"
	UNIT_Gigabytes_Second = "Gigabytes/Second"
	UNIT_Terabytes_Second = "Terabytes/Second"
	UNIT_Bits_Second      = "Bits/Second"
	UNIT_Kilobits_Second  = "Kilobits/Second"
	UNIT_Megabits_Second  = "Megabits/Second"
	UNIT_Gigabits_Second  = "Gigabits/Second"
	UNIT_Terabits_Second  = "Terabits/Second"
	UNIT_Count_Second     = "Count/Second"
	UNIT_None             = "None"
)

type CloudWatch interface {
	PutMetric(m *Metric)
}

type Metric struct {
	Name       string
	Value      float64
	Unit       string
	Statistic  *Statistic
	dimensions []*cwpkg.Dimension
}

type Statistic struct {
	// The maximum value of the sample set.
	Maximum float64 `type:"double" required:"true"`

	// The minimum value of the sample set.
	Minimum float64 `type:"double" required:"true"`

	// The number of samples used for the statistic set.
	SampleCount float64 `type:"double" required:"true"`

	// The sum of values for the sample set.
	Sum float64 `type:"double" required:"true"`
}

func New(logger *log.SugaredLogger) CloudWatch {
	session := awsconfig.NewSession()
	cw := &cloudwatch{
		namespace: "VHS",
		client:    cwpkg.New(session),
		ec2:       ec2metadata.New(session),
		logger:    logger,
	}
	cw.LoadInstanceDimensions()
	return cw
}

type cloudwatch struct {
	namespace          string
	client             *cwpkg.CloudWatch
	ec2                *ec2metadata.EC2Metadata
	instanceDimensions []*cwpkg.Dimension
	logger             *log.SugaredLogger
}

func (cw *cloudwatch) PutMetric(m *Metric) {
	if m.dimensions != nil {
		m.dimensions = append(cw.instanceDimensions, m.dimensions...)
	} else {
		m.dimensions = cw.instanceDimensions
	}

	_, err := cw.client.PutMetricData(&cwpkg.PutMetricDataInput{
		Namespace: aws.String(cw.namespace),
		MetricData: []*cwpkg.MetricDatum{
			&cwpkg.MetricDatum{
				Dimensions:      m.dimensions,
				StatisticValues: m.getStatisticSet(),
				MetricName:      aws.String(m.Name),
				Value:           aws.Float64(m.Value),
				Unit:            aws.String(m.Unit),
				Timestamp:       aws.Time(time.Now()),
			},
		},
	})

	if err != nil {
		cw.logger.Errorf("Unable to post metric: %v", err)
	}
}

func (cw *cloudwatch) LoadInstanceDimensions() {
	cw.instanceDimensions = []*cwpkg.Dimension{}

	envName := os.Getenv("ENV_NAME")
	if len(envName) > 0 {
		cw.instanceDimensions = append(cw.instanceDimensions, &cwpkg.Dimension{
			Name:  aws.String("EnvironmentName"),
			Value: aws.String(envName),
		})
	}
}

func (m *Metric) AddDimension(key, value string) {
	if m.dimensions == nil {
		m.dimensions = []*cwpkg.Dimension{}
	}

	m.dimensions = append(m.dimensions, &cwpkg.Dimension{
		Name:  aws.String(key),
		Value: aws.String(value),
	})
}

func (m *Metric) getStatisticSet() *cwpkg.StatisticSet {
	if m.Statistic != nil {
		return &cwpkg.StatisticSet{
			Maximum:     aws.Float64(m.Statistic.Maximum),
			Minimum:     aws.Float64(m.Statistic.Minimum),
			SampleCount: aws.Float64(m.Statistic.SampleCount),
			Sum:         aws.Float64(m.Statistic.Sum),
		}
	}
	return nil
}
