package s3

import (
	"fmt"
	"io/ioutil"
	"os"

	"github.com/aws/aws-sdk-go/aws"
	s3pkg "github.com/aws/aws-sdk-go/service/s3"
)

const PUBLIC_READ_URI = "http://acs.amazonaws.com/groups/global/AllUsers"

func (s *s3) ObjectExists(key string) (bool, error) {
	input := &s3pkg.HeadObjectInput{
		Bucket: aws.String(s.Bucket),
		Key:    aws.String(key),
	}

	_, err := s.client.HeadObject(input)
	if err != nil {
		return false, err
	}
	return true, nil
}

func (s *s3) CacheObject(key, path string) error {
	if _, err := os.Stat(path); os.IsNotExist(err) {
		data, err := s.GetObject(key)
		s.logger.Log("Cached s3://", s.Bucket, "/", key, " to ", path)
		if err != nil {
			return err
		}
		if err := ioutil.WriteFile(path, data, 0644); err != nil {
			return err
		}
	} else {
		s.logger.Log("Object already cached: ", path)
	}
	return nil
}

func (s *s3) GetObject(key string) ([]byte, error) {
	input := &s3pkg.GetObjectInput{
		Bucket: aws.String(s.Bucket),
		Key:    aws.String(key),
	}
	resp, err := s.client.GetObject(input)
	if err != nil {
		err = fmt.Errorf("Error retrieving object from s3; bucket=%v, key=%v, err=%v\n", s.Bucket, key, err)
		return nil, err
	}
	defer resp.Body.Close()
	s.logger.Log("Retrieved object s3://", s.Bucket, "/", key)
	return ioutil.ReadAll(resp.Body)
}
