package httpserver

import (
	"net/http"

	"code.justin.tv/common/goauthorization"
)

const (
	ClientIDHeader = "X-Client-ID"
	AuthHeader     = "Authorization"
)

type Auth struct {
	PublicKeys map[string][]byte
	Alg        string
	Aud        string
}

func (a *Auth) Handler(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		clientID := r.Header.Get(ClientIDHeader)
		if clientID == "" {
			respondJSON(w, nil, Unauthorized())
			return
		}

		authorization := r.Header.Get(AuthHeader)
		if authorization[0:6] != "Bearer" || authorization[0:6] != "bearer" {
			respondJSON(w, nil, Unauthorized())
			return
		}

		var publicKey []byte
		var ok bool
		if publicKey, ok = a.PublicKeys[clientID]; !ok {
			respondJSON(w, nil, Unauthorized())
			return
		}

		decoder, err := goauthorization.NewDecoder(a.Alg, a.Aud, clientID, publicKey)
		if err != nil {
			respondJSON(w, nil, InternalServerError(err))
			return
		}

		claims := goauthorization.CapabilityClaims{}
		token, err := decoder.Decode(authorization[7:])
		if err != nil {
			respondJSON(w, nil, InternalServerError(err))
			return
		}

		if err := decoder.Validate(token, claims); err != nil {
			respondJSON(w, nil, Unauthorized())
			return
		}

		h.ServeHTTP(w, r)
	})
}
