package main

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"
	"time"

	"code.justin.tv/vodsvc/aws/awsconfig"
	"code.justin.tv/vodsvc/httpserver"
	"code.justin.tv/vodsvc/sts/src/probe"
	"code.justin.tv/vodsvc/sts/src/transcoder"
	"code.justin.tv/vodsvc/vhs/src/dynamodb"
	"code.justin.tv/vodsvc/vhs/src/types"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
	log "go.uber.org/zap"
)

const sourceFile = "source"

var logger *log.SugaredLogger
var ddb dynamodb.UploadTable

func main() {
	logger = httpserver.DefaultLogger().Sugar()
	uploadId := parseArgs()

	u, err := getUpload(uploadId)
	if err != nil {
		fatal(uploadId, err)
	}

	workDir, source, err := prepareDir(uploadId)
	defer os.RemoveAll(workDir)
	if err != nil {
		failErr(u, err)
	}

	if err := download(u, source); err != nil {
		failErr(u, err)
	}

	md, terr := validate(source)
	if terr != nil {
		fail(u, terr)
	}

	video, err := transcode(workDir, u, source, md)
	if err != nil {
		failErr(u, err)
	}

	if err := upload(u, workDir); err != nil {
		failErr(u, err)
	}

	succeed(u, video)
}

func parseArgs() (uploadId string) {
	if len(os.Args) < 3 {
		logger.Info("USAGE: ", filepath.Base(os.Args[0]), " <uploadId> <uploadTable>")
		logger.Fatal(newError("'uploadId' and 'uploadTable' are required arguments"))
	}

	uploadId = os.Args[1]
	uploadTable := os.Args[2]

	ddb = dynamodb.New(uploadTable, logger)

	return uploadId
}

func getUpload(id string) (types.Upload, error) {
	var u = types.Upload{}
	u, err := ddb.GetUpload(id)
	if u.Id == "" && err == nil {
		err = errors.New("Upload not found; id=" + id)
	}
	if err != nil {
		return u, newError("Failed getting Upload from DDB; id=", id, " err=", err.Error())
	}
	logger.Infow("Retrieved item from DDB", "id", id)
	return u, nil
}

func prepareDir(id string) (string, *os.File, error) {
	dir, err := ioutil.TempDir("", id)
	if err != nil {
		return dir, nil, newError("Failed creating temp dir; err=", err.Error())
	}

	file, err := os.Create(filepath.Join(dir, sourceFile))
	if err != nil {
		return dir, nil, newError("Failed creating target source file; err=", err.Error())
	}

	return dir, file, nil
}

func download(upload types.Upload, target *os.File) error {
	downloader := s3manager.NewDownloader(awsconfig.Session)
	sourceKey := filepath.Join(upload.S3Prefix, upload.SourceFile)
	uri := s3uri(upload.S3Bucket, sourceKey)
	startTime := time.Now()
	logger.Infow("Downloading source",
		"uri", uri,
		"target", target.Name(),
		"partSize", downloader.PartSize,
		"concurrency", downloader.Concurrency)

	input := &s3.GetObjectInput{
		Bucket: aws.String(upload.S3Bucket),
		Key:    aws.String(filepath.Join(upload.S3Prefix, upload.SourceFile)),
	}

	bytes, err := downloader.Download(target, input)
	if err != nil {
		return newError("Failed downloading from S3; id=", upload.Id, " err=", err.Error())
	}
	seconds := time.Since(startTime).Seconds()
	mbSec := float64(bytes) / (1024 * 1024) / seconds
	logger.Infow("Finished downloading source",
		"uri", uri,
		"bytes", bytes,
		"seconds", seconds,
		"MB/s", mbSec)
	return nil
}

func upload(upload types.Upload, dir string) error {
	uploader := s3manager.NewUploader(awsconfig.Session)
	files, err := getAllFiles(dir)
	if err != nil {
		return newError("Error walking directory tree: ", dir, err.Error())
	}

	startTime := time.Now()
	logger.Infow("Uploading video files",
		"s3bucket", upload.VideoBucket,
		"s3prefix", upload.S3Prefix,
		"partSize", uploader.PartSize,
		"concurrency", uploader.Concurrency)

	objects := make([]s3manager.BatchUploadObject, len(files))
	for i, f := range files {
		if f == upload.SourceFile {
			continue
		}

		file, err := os.Open(f)
		if err != nil {
			return newError("Error opening video file: ", f, ", err=", err.Error())
		}
		input := s3manager.UploadInput{
			Bucket: aws.String(upload.VideoBucket),
			Key:    aws.String(filepath.Join(upload.S3Prefix, strings.TrimPrefix(f, dir))),
			Body:   file,
		}
		objects[i] = s3manager.BatchUploadObject{Object: &input}
	}
	iter := &s3manager.UploadObjectsIterator{Objects: objects}
	err = uploader.UploadWithIterator(aws.BackgroundContext(), iter)
	if err != nil {
		return newError("Failed uploading to S3; id=", upload.Id, " err=", err.Error())
	}

	seconds := time.Since(startTime).Seconds()
	logger.Infow("Finished uploading video files",
		"objects", len(objects),
		"seconds", seconds)

	return nil
}

func validate(source *os.File) (probe.VideoMetadata, *transcoder.Error) {
	metadata := probe.VideoMetadata{}
	// See VULN-134 for why we prevent m3u files from being uploaded.
	// They allow arbitrary URLs to be called when ffprobe/ffmpeg is run.
	const m3uTag = "#EXTM3U"
	buf := make([]byte, len(m3uTag))
	_, err := source.ReadAt(buf, 0)
	if err != nil {
		return metadata, transcoder.NewErrInternalFailure("Failed reading source video: " + err.Error())
	}

	if bytes.Equal(buf, []byte(m3uTag)) {
		return metadata, transcoder.NewErrUnsupportedFormat("m3u files are not supported")
	}

	metadata, terr := probe.Probe(source.Name())
	if terr != nil {
		return metadata, terr
	}

	if len(metadata.VideoStreams) == 0 {
		return metadata, transcoder.NewErrBadSource("No video streams found in source")
	}

	codec := metadata.VideoStreams[0].Codec
	if codec != probe.H264 {
		return metadata, transcoder.NewErrUnsupportedFormat("Video codec not supported: " + string(codec))
	}
	return metadata, nil
}

func transcode(workDir string, upload types.Upload, source *os.File, md probe.VideoMetadata) (transcoder.Video, error) {
	conf := transcoder.ConfigWithDir(transcoder.DefaultTranscoderConfig, workDir)
	conf.Transcodes = transcoder.BitrateFilter(md.VideoStreams[0].BitRate, conf.Transcodes)

	return transcoder.Transcode(upload.Id, source, conf)
}

func uploadManifest(u types.Upload, v transcoder.Video) error {
	sb := &strings.Builder{}
	enc := json.NewEncoder(sb)
	enc.SetIndent("", "  ")

	if err := enc.Encode(v); err != nil {
		return fmt.Errorf("Error encoding video into json: " + err.Error())
	}

	s3c := s3.New(awsconfig.Session)
	input := &s3.PutObjectInput{
		Body:   aws.ReadSeekCloser(strings.NewReader(sb.String())),
		Bucket: aws.String(u.VideoBucket),
		Key:    aws.String(filepath.Join(u.S3Prefix, "video.json")),
	}

	_, err := s3c.PutObject(input)
	if err != nil {
		return fmt.Errorf("Error writing manifest to S3: " + err.Error())
	}

	return nil
}

func getAllFiles(dir string) ([]string, error) {
	files := []string{}
	err := filepath.Walk(dir, func(path string, info os.FileInfo, err error) error {
		if !info.IsDir() {
			files = append(files, path)
		}
		return nil
	})
	return files, err
}

func s3uri(bucket string, key string) string {
	return "s3://" + bucket + "/" + key
}

func failErr(u types.Upload, err error) {
	fail(u, transcoder.NewErrInternalFailure(err.Error()))
}

func fail(u types.Upload, terr *transcoder.Error) {
	v := transcoder.Video{
		Id:            u.Id,
		Status:        transcoder.StatusFailure,
		FailureCode:   terr.Code,
		FailureReason: terr.Message,
	}

	if err := uploadManifest(u, v); err != nil {
		fatal(u.Id, err)
	}

	logger.Infow("Failing transcode", "id", u.Id, "code", terr.Code, "reason", terr.Message)
	os.Exit(0)
}

func succeed(u types.Upload, v transcoder.Video) {
	v.Status = transcoder.StatusSuccess
	if err := uploadManifest(u, v); err != nil {
		fatal(u.Id, err)
	}

	logger.Infow("Transcode success", "id", u.Id)
	os.Exit(0)
}

func fatal(id string, err error) {
	logger.Fatalw("Transcode error", "id", id, "err", err)
}

func newError(m ...string) error {
	return errors.New(fmt.Sprint(m))
}
