package transcoder

import (
	"bufio"
	"encoding/json"
	"fmt"
	"os"
	"os/exec"
	"path"
	"path/filepath"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/twitchtranscoder"
	"code.justin.tv/vodsvc/httpserver"
	log "go.uber.org/zap"
)

type Output struct {
	Segments   []avdata.Segment
	Codecs     []avdata.Codec
	Thumbnails []avdata.Thumbnail
}

var logger *log.SugaredLogger

const twitchtranscoderBin = "/usr/local/bin/TwitchTranscoder"

func init() {
	logger = httpserver.DefaultLogger().Sugar()
}

// Run the TwitchTranscoder binary with the supplied config.
func Transcode(id string, source *os.File, conf TranscoderConfig) (Video, error) {

	var video = Video{}

	confFile, err := writeConfig(conf)
	if err != nil {
		return video, fmt.Errorf("unable to write input config to disk: %s", err)
	}
	logger.Info("Wrote config file: " + confFile.Name())

	if err := prepareOutputDirs(conf); err != nil {
		return video, fmt.Errorf("unable to prepare output directories: %s", err)
	}
	logger.Info("Prepared output directories")

	cmd, stderr, err := prepareCmd(twitchtranscoderBin, source, confFile, conf.FileBase)
	defer stderr.Close()
	if err != nil {
		return video, fmt.Errorf("error preparing run command: %s", err)
	}
	logger.Infow("Prepared command", "path", cmd.Path, "args", cmd.Args)

	if err := run(cmd, stderr); err != nil {
		return video, fmt.Errorf("transcoder binary exited with non-zero code: %s", err)
	}

	output := parseStderr(stderr)

	video, err = toManifest(id, conf, output)
	if err != nil {
		return video, err
	}

	return video, nil
}

// Writes the current transcoder config to a tempfile. It's the
// caller's responsibility to remove the file when done with it.
func writeConfig(c TranscoderConfig) (f *os.File, err error) {
	f, err = os.Create(filepath.Join(c.FileBase, "config.json"))
	if err != nil {
		return nil, err
	}

	err = json.NewEncoder(f).Encode(c)
	if err != nil {
		return nil, err
	}

	err = f.Close()
	if err != nil {
		return nil, err
	}
	return f, nil
}

// twitchtranscoder emits a lot of messages to stderr. Some of those
// messages describe the segments that it writes to disk. We want to
// scoop up and parse those messages.
func parseStderr(stderr *os.File) (output *Output) {
	output = new(Output)
	scanner := bufio.NewScanner(stderr)
	for scanner.Scan() {
		parsed, _ := twitchtranscoder.ParseLine(scanner.Bytes())

		switch val := parsed.(type) {
		case avdata.Codec:
			output.Codecs = append(output.Codecs, val)
		case avdata.Segment:
			output.Segments = append(output.Segments, val)
		case avdata.Thumbnail:
			val.Path = path.Base(val.Path)
			output.Thumbnails = append(output.Thumbnails, val)
		}
	}
	return output
}

func run(cmd *exec.Cmd, stderr *os.File) error {
	if err := cmd.Run(); err != nil {
		return err
	}
	if err := stderr.Sync(); err != nil {
		return err
	}
	if _, err := stderr.Seek(0, 0); err != nil {
		return err
	}
	return nil
}

func prepareOutputDirs(c TranscoderConfig) (err error) {
	mode := os.FileMode(0700)
	err = os.Mkdir(c.AudioOnly.FileBase, mode)
	err = os.Mkdir(c.Thumbnail.FileBase, mode)
	for _, r := range c.Transcodes {
		err = os.Mkdir(r.FileBase, mode)
	}
	return
}

// prepareCmd sets up the command to be executed by the runner, and also
// returns the buffer that stderr will get written into.
func prepareCmd(bin string, source *os.File, conf *os.File, workDir string) (*exec.Cmd, *os.File, error) {
	cmd := exec.Command(
		bin,
		"--encoder_config", conf.Name(),
		"--inputFile", source.Name(),
		"--disable_listen_stdin", "true",
		// Setting the encoder_threads to 0 lets the H264 encoder use as
		// many threads as it wants. Since we only transcode one input at
		// a time, this is the right value.
		"--encoder_threads", "0",
	)
	stderr, err := os.Create(filepath.Join(workDir, "transcoder.log"))
	cmd.Stderr = stderr
	return cmd, stderr, err
}
