package jwt

// Taken from https://github.com/dgrijalva/jwt-go/blob/master/ecdsa_test.go
import (
	"encoding/base64"
	"strings"
	"testing"
)

var ecdsaTestData = []struct {
	name        string
	privateKey  string
	publicKey   string
	tokenString string
	alg         string
	claims      map[string]interface{}
	valid       bool
}{
	{
		"Basic ES256",
		"_key/ec256-private.pem",
		"_key/ec256-public.pem",
		"eyJ0eXAiOiJKV1QiLCJhbGciOiJFUzI1NiJ9.eyJmb28iOiJiYXIifQ.feG39E-bn8HXAKhzDZq7yEAPWYDhZlwTn3sePJnU9VrGMmwdXAIEyoOnrjreYlVM_Z4N13eK9-TmMTWyfKJtHQ",
		"ES256",
		map[string]interface{}{"foo": "bar"},
		true,
	},
	{
		"Basic ES384",
		"_key/ec384-private.pem",
		"_key/ec384-public.pem",
		"eyJ0eXAiOiJKV1QiLCJhbGciOiJFUzM4NCJ9.eyJmb28iOiJiYXIifQ.ngAfKMbJUh0WWubSIYe5GMsA-aHNKwFbJk_wq3lq23aPp8H2anb1rRILIzVR0gUf4a8WzDtrzmiikuPWyCS6CN4-PwdgTk-5nehC7JXqlaBZU05p3toM3nWCwm_LXcld",
		"ES384",
		map[string]interface{}{"foo": "bar"},
		true,
	},
	{
		"Basic ES512",
		"_key/ec512-private.pem",
		"_key/ec512-public.pem",
		"eyJ0eXAiOiJKV1QiLCJhbGciOiJFUzUxMiJ9.eyJmb28iOiJiYXIifQ.AAU0TvGQOcdg2OvrwY73NHKgfk26UDekh9Prz-L_iWuTBIBqOFCWwwLsRiHB1JOddfKAls5do1W0jR_F30JpVd-6AJeTjGKA4C1A1H6gIKwRY0o_tFDIydZCl_lMBMeG5VNFAjO86-WCSKwc3hqaGkq1MugPRq_qrF9AVbuEB4JPLyL5",
		"ES512",
		map[string]interface{}{"foo": "bar"},
		true,
	},
	{
		"basic ES256 invalid: foo => bar",
		"_key/ec256-private.pem",
		"_key/ec256-public.pem",
		"eyJhbGciOiJFUzI1NiIsInR5cCI6IkpXVCJ9.eyJmb28iOiJiYXIifQ.MEQCIHoSJnmGlPaVQDqacx_2XlXEhhqtWceVopjomc2PJLtdAiAUTeGPoNYxZw0z8mgOnnIcjoxRuNDVZvybRZF3wR1l8W",
		"ES256",
		map[string]interface{}{"foo": "bar"},
		false,
	},
}

func TestECDSAVerify(t *testing.T) {
	for _, data := range ecdsaTestData {
		var err error

		parts := strings.Split(data.tokenString, ".")
		key := GetECDSAPrivateKey(data.privateKey, data.publicKey)
		alg := eccAlgs[data.alg](key)

		sig, _ := DecodeSegment(parts[2])
		err = alg.Validate([]byte(strings.Join(parts[0:2], ".")), sig)
		if data.valid && err != nil {
			t.Errorf("[%v] Error while verifying key: %v", data.name, err)
		}
		if !data.valid && err == nil {
			t.Errorf("[%v] Invalid key passed validation", data.name)
		}
	}
}

func TestECDSASign(t *testing.T) {
	for _, data := range ecdsaTestData {
		if data.valid {
			parts := strings.Split(data.tokenString, ".")
			key := GetECDSAPrivateKey(data.privateKey, data.publicKey)
			alg := eccAlgs[data.alg](key)

			body := EncodeSegment(strings.Join(parts[0:2], "."))
			sig, err := alg.Sign(body)
			if err != nil {
				t.Errorf("[%v] Error signing token: %v", data.name, err)
			}
			if string(sig) == parts[2] {
				t.Errorf("[%v] Identical signatures\nbefore:\n%v\nafter:\n%v", data.name, parts[2], sig)
			}
		}
	}
}

// Encode JWT specific base64url encoding with padding stripped
func EncodeSegment(seg string) []byte {
	bytes := []byte(seg)
	encoded := strings.TrimRight(base64.URLEncoding.EncodeToString(bytes), "=")
	return []byte(encoded)
}

// Decode JWT specific base64url encoding with padding stripped
func DecodeSegment(seg string) ([]byte, error) {
	if l := len(seg) % 4; l > 0 {
		seg += strings.Repeat("=", 4-l)
	}

	return base64.URLEncoding.DecodeString(seg)
}
