package main

import (
	"encoding/json"
	"fmt"
	"io"
	"log"
	"net"
	"net/http"
	"text/template"

	"github.com/gorilla/mux"
	"golang.org/x/net/http2"

	"code.justin.tv/video/gotranscoder/pkg/config"
	"code.justin.tv/video/gotranscoder/pkg/usher"
)

type homeInfo struct {
	User      string
	RtmpURL   string
	Qualities []string
}

type channelPlayer struct {
	Channel string
	Quality string
}

type confInfo struct {
	GotranscoderConfig config.Values
	EncoderConfig      *EncoderConfig
	StreamInfo         *usher.StreamInfo
	UsherTranscode     *usher.HlsTranscode
}

// HomeHandler to display information on the current transcode process
func HomeHandler(w http.ResponseWriter, _ *http.Request) {
	t, err := template.ParseFiles("./static/templates/debug.html")
	if err != nil {
		log.Println("Error getting template for main page", err)
	}

	info := homeInfo{
		User:      *channel,
		RtmpURL:   encoderConfig.RtmpURL,
		Qualities: outputQualities,
	}

	err = t.Execute(w, info)
	if err != nil {
		log.Println("Error rendering template for main page", err)
	}
}

// DebugHandler to display information on the current transcode process
func DebugHandler(w http.ResponseWriter, _ *http.Request) {
	_, err := w.Write([]byte("TODO: add chooser to playback for different qualities and show debug information"))
	if err != nil {
		log.Println("Error writing DebugHandler response", err)
	}
}

// ConfHandler to display the current running configuration
func ConfHandler(w http.ResponseWriter, _ *http.Request) {
	configs := confInfo{
		GotranscoderConfig: cfg,
		EncoderConfig:      encoderConfig,
		StreamInfo:         stream,
		UsherTranscode:     usherTranscode,
	}

	err := json.NewEncoder(w).Encode(configs)
	if err != nil {
		log.Println("Error writing config to response")
	}
}

// PlaybackHandler renders an HTML5 basic player and plays back the content
// generated by this process
func PlaybackHandler(w http.ResponseWriter, r *http.Request) {
	t, err := template.ParseFiles("./static/templates/player.html")
	if err != nil {
		log.Println("Error getting template for debug player", err)
	}

	vars := mux.Vars(r)
	if val, ok := vars["quality"]; ok {
		s := channelPlayer{
			Channel: *channel,
			Quality: val,
		}

		if err := t.Execute(w, s); err != nil {
			log.Println("Failed to execute player template", err)
		}
	} else {
		w.WriteHeader(http.StatusBadRequest)
		w.Header().Set("Content-Type", "text/plain; charset=utf-8")
		_, err := io.WriteString(w, "_quality_ argument not specified in the URL.")
		if err != nil {
			log.Println("Error writing PlaybackHandler response", err)
		}
	}
}

// initHTTP wil start an http daemon with https/2
func initHTTP() (int, error) {
	var srv http.Server
	// bind to random available port
	srv.Addr = ":0"
	if err := http2.ConfigureServer(&srv, nil); err != nil {
		log.Fatalln("Failed to configure http/2 server for rpcs", err)
	}

	log.Printf("Serving chunks from %s ", transcodesBasePath)

	r := mux.NewRouter()
	r.HandleFunc("/", HomeHandler)
	r.HandleFunc("/debug", DebugHandler)
	r.HandleFunc("/debug/conf", ConfHandler)
	r.HandleFunc("/playback/{quality}", PlaybackHandler)
	//r.HandleFunc("/webstatic/", StaticHandler)

	// Serve generated chunk files via http2
	http.Handle("/transcode/", http.StripPrefix(
		"/transcode/",
		http.FileServer(
			http.Dir(fmt.Sprintf("%s/", transcodesBasePath)),
		),
	))

	// Serve static helper files used for the web debug page
	http.Handle("/webstatic/", http.StripPrefix(
		"/webstatic/",
		http.FileServer(
			http.Dir(fmt.Sprintf("./static/files/")),
		),
	))

	http.Handle("/", r)
	// Spin a listener on port 0, get a random assigned port, and return it.
	// Each gotranscoder process will be listening on an individual port
	// servingdata
	ln, err := net.Listen("tcp", ":0")
	if err != nil {
		log.Println("Error listening to debug socket")
		return 0, err
	}

	port := ln.Addr().(*net.TCPAddr).Port

	log.Printf("Starting Gotranscoder HTTP listener on port: %d", port)

	go func() {
		err := srv.Serve(ln)
		if err != nil {
			log.Println("Gotranscoder HTTP server exited with", err)
		}
	}()
	return port, nil
}
