package main

import (
	"errors"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"golang.org/x/net/context"

	"code.justin.tv/video/gotranscoder/pkg/m3u8"
	"code.justin.tv/video/gotranscoder/pkg/tenfoot"
	"code.justin.tv/video/origin/rpc/originctl"
	"code.justin.tv/video/origin/rpc/originctl/originctlfakes"
)

func TestInitOriginSession(t *testing.T) {
	fakeOrigin := &originctlfakes.FakeOriginControlV1{}
	s := httptest.NewServer(originctl.NewOriginControlV1Server(fakeOrigin, nil, nil))
	defer s.Close()
	endpoint := s.URL

	t.Run("happy case", func(t *testing.T) {
		// onExit is a required global for initOriginSession to work.
		onExit = new(OnExit)
		cleanupInterval = 0

		fakeOrigin.StartSessionReturns(&originctl.StartSessionResponse{
			Session: &originctl.Session{Id: "test-session"},
		}, nil)

		// Create an origin session.
		session, callBack, err := initOriginSession(endpoint, 1, []string{"source"})
		require.NoError(t, err)

		onExit.Add(callBack)

		// We should have made a call to the origin service. We should have the
		// expected session.
		assert.Equal(t, 1, fakeOrigin.StartSessionCallCount())
		require.NotNil(t, session)
		assert.Equal(t, "test-session", session.Id)

		// A TerminateSession callback should have been added to the onExit global.
		assert.Len(t, onExit.callbacks, 1)

		terminateCalled := make(chan bool)
		fakeOrigin.TerminateSessionStub = func(context.Context, *originctl.TerminateSessionRequest) (*originctl.TerminateSessionResponse, error) {
			close(terminateCalled)
			return &originctl.TerminateSessionResponse{}, nil
		}
		onExit.Cleanup()
		select {
		case <-time.After(time.Second):
			t.Errorf("terminate wasn't called in time")
		case <-terminateCalled:
		}
	})

	t.Run("erroring originctl", func(t *testing.T) {
		onExit = new(OnExit)
		fakeOrigin.StartSessionReturns(nil, errors.New("everything sucks"))
		_, _, err := initOriginSession(endpoint, 1, []string{"source"})
		require.Error(t, err)
		// We shouldn't have a TerminateSession callback if we couldn't set up a
		// session.
		assert.Len(t, onExit.callbacks, 0)
	})
}

func TestInitOriginFoot(t *testing.T) {
	// Set the 'sessionID' and 'cfg.Segment.TargetDuration' globals, but return a
	// func which can be called to reset them to their earlier values. These
	// globals are needed when calling initOriginFoot.
	setGlobals := func(_sessionID, targetDuration string) (reset func()) {
		oldSessionID := sessionID
		oldTargetDuration := cfg.Segment.TargetDuration

		reset = func() {
			sessionID = oldSessionID
			cfg.Segment.TargetDuration = oldTargetDuration
		}

		sessionID = _sessionID
		cfg.Segment.TargetDuration = targetDuration

		return reset
	}

	const goodUUID = "ad3c3566-9e59-41b4-8d5d-6a236cb4e217"

	t.Run("happy path", func(t *testing.T) {
		reset := setGlobals(goodUUID, "2")
		defer reset()

		sh := &SegmentHandler{tenfootGlue: tenfoot.Noop()}
		session := &originctl.Session{Id: "1"}
		require.NoError(t, initOriginFoot(sh, "endpoint", session, time.Now()))

		assert.NoError(t, sh.tenfootGlue.Wait())
		assert.NoError(t, sh.tenfootGlue.PostSegment(nil))
		assert.NoError(t, sh.tenfootGlue.PostPlaylist(m3u8.Playlist{}, "", "", "", nil, nil))

		// the sh.tenfootGlue should now be a *tenfoot.OriginFoot
		_, ok := sh.tenfootGlue.(*tenfoot.OriginFoot)
		assert.True(t, ok)
	})

	t.Run("nil session", func(t *testing.T) {
		sh := &SegmentHandler{tenfootGlue: tenfoot.Noop()}

		err := initOriginFoot(nil, "", nil, time.Now())
		require.Error(t, err)
		assert.Equal(t, "session not established with origin", err.Error())

		assert.NoError(t, sh.tenfootGlue.Wait())
		assert.NoError(t, sh.tenfootGlue.PostSegment(nil))
		assert.NoError(t, sh.tenfootGlue.PostPlaylist(m3u8.Playlist{}, "", "", "", nil, nil))
	})

	t.Run("bad session ID", func(t *testing.T) {
		reset := setGlobals("bogus", "2")
		defer reset()
		sh := &SegmentHandler{tenfootGlue: tenfoot.Noop()}
		session := &originctl.Session{Id: "1"}

		err := initOriginFoot(sh, "endpoint", session, time.Now())

		assert.Error(t, err)
		assert.NoError(t, sh.tenfootGlue.Wait())
		assert.NoError(t, sh.tenfootGlue.PostSegment(nil))
		assert.NoError(t, sh.tenfootGlue.PostPlaylist(m3u8.Playlist{}, "", "", "", nil, nil))
	})

	t.Run("bad target duration", func(t *testing.T) {
		reset := setGlobals(goodUUID, "sandwiches")
		defer reset()

		sh := &SegmentHandler{tenfootGlue: tenfoot.Noop()}
		session := &originctl.Session{Id: "1"}

		err := initOriginFoot(sh, "endpoint", session, time.Now())

		assert.Error(t, err)
		assert.NoError(t, sh.tenfootGlue.Wait())
		assert.NoError(t, sh.tenfootGlue.PostSegment(nil))
		assert.NoError(t, sh.tenfootGlue.PostPlaylist(m3u8.Playlist{}, "", "", "", nil, nil))
	})
}

func TestS3VodUrlValue(t *testing.T) {
	s3VodUrl := getS3VodUrl("test-bucket", "vod-key")
	require.Equal(t, "https://test-bucket.s3.amazonaws.com/vod-key", s3VodUrl)
}

func TestRtmpUrlLowLatencyDarklaunch(t *testing.T) {
	setGlobals := func(_rtmpEndpoint, _channel string) (reset func()) {
		oldRtmpEndpoint := *rtmpEndpoint
		oldChannel := *channel

		reset = func() {
			*rtmpEndpoint = oldRtmpEndpoint
			*channel = oldChannel
		}

		*rtmpEndpoint = _rtmpEndpoint
		*channel = _channel

		return reset
	}

	reset := setGlobals("rtmp://10.23.50.160:18747/app", "lvs.twitch-lowlatency-darklaunch.testchannel")
	rtmpUrl := getRtmpURL()
	assert.Equal(t, "rtmp://10.23.50.160:18747/app/live_user_testchannel", rtmpUrl)
	reset()

	// Test other LVS channels
	reset = setGlobals("rtmp://10.23.50.160:18747/app", "lvs.amazonlivetools.testchannel")
	rtmpUrl = getRtmpURL()
	assert.Equal(t, "rtmp://10.23.50.160:18747/app/live_user_lvs.amazonlivetools.testchannel", rtmpUrl)
	reset()

	// Test  twitch channel
	reset = setGlobals("rtmp://10.23.50.160:18747/app", "testchannel")
	rtmpUrl = getRtmpURL()
	assert.Equal(t, "rtmp://10.23.50.160:18747/app/live_user_testchannel", rtmpUrl)
	reset()
}

func TestFullRtmpUrl(t *testing.T) {
	setGlobals := func(_rtmpUrl, _rtmpEndpoint, _channel string) (reset func()) {
		oldRtmpUrl := _rtmpUrl
		oldRtmpEndpoint := *rtmpEndpoint
		oldChannel := *channel

		reset = func() {
			*rtmpEndpoint = oldRtmpEndpoint
			*channel = oldChannel
			*rtmpUrl = oldRtmpUrl
		}

		*rtmpEndpoint = _rtmpEndpoint
		*channel = _channel
		*rtmpUrl = _rtmpUrl

		return reset
	}

	reset := setGlobals("rtmp://10.23.50.160:18747/app_live_user_test", "rtmp://10.23.50.160:18747/app", "lvs.twitch-lowlatency-darklaunch.testchannel")
	rtmpUrl := getRtmpURL()
	assert.Equal(t, "rtmp://10.23.50.160:18747/app_live_user_test", rtmpUrl)
	reset()

}
