package main

import (
	"sync"
	"time"
)

// Time between calling each cleanup function
var cleanupInterval = 16 * time.Second

// OnExit stores a slice of callbacks to call
// when we "Cleanup()"
type OnExit struct {
	mut       sync.Mutex
	callbacks []func()
}

// Add a 'fn' to our slice of callbacks
func (e *OnExit) Add(fn func()) {
	e.mut.Lock()
	e.callbacks = append(e.callbacks, fn)
	e.mut.Unlock()
}

// Cleanup runs all the callbacks
func (e *OnExit) Cleanup() {
	e.mut.Lock()
	for i, fn := range e.callbacks {
		go fn()
		time.Sleep(cleanupInterval)
		e.callbacks[i] = nil
	}
	e.callbacks = e.callbacks[:0]
	e.mut.Unlock()
}
