package main

import (
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"path/filepath"
	"time"

	"github.com/cenkalti/backoff"

	"code.justin.tv/video/gotranscoder/pkg/notify"
	"code.justin.tv/video/gotranscoder/pkg/statsd"
	"code.justin.tv/video/gotranscoder/pkg/usher"
)

// TranscodeAvailable loads the Transcode profiles and transcode quality definitions
func TranscodeAvailable(outputPath string, qualities []string) bool {
	var ready int
	for _, q := range qualities {
		log.Println("[READY] Checking readiness of: ", filepath.Join(outputPath, q))
		files, _ := ioutil.ReadDir(filepath.Join(outputPath, q))
		if len(files) > 4 {
			ready++
		} else if q == "audio_only" { //dont block marking the stream ready if audio channel is absent
			ready++
		}
	}

	if ready < len(qualities) {
		return false
	}

	return true
}

// MarkReady in usher
func MarkReady(isLvsStream bool) error {
	exp := backoff.NewExponentialBackOff()
	exp.MaxElapsedTime = 20 * time.Second
	var operation backoff.Operation

	if isLvsStream {
		operation = func() error {
			return usherCfg.MarkLvsStreamReady(usherTranscode.Destination)
		}
	}

	if !isLvsStream {
		operation = func() error {
			return usherCfg.MarkHlsStreamReady(usherTranscode.Destination)
		}
	}

	err := backoff.Retry(operation, exp)
	if err != nil {
		log.Printf("[READY] failed to mark stream as ready %s\n", err)
		statsd.Inc(constTranscodeReadyFail, 1, 1.0)
		return err
	}

	log.Println("[READY] Stream marked as ready")
	statsd.Inc(constTranscodeReady, 1, 1.0)
	return nil
}

// ReadyLoop Scans the transcoder output every half second and mark as ready once all qualities have  an output
func ReadyLoop(outputPath string, qualities []string, notifyObj notify.Notifier) {
	exp := backoff.NewExponentialBackOff()
	exp.MaxElapsedTime = 30 * time.Second

	operation := func() error {

		if !TranscodeAvailable(outputPath, qualities) {
			return errors.New("Transcode not ready")
		}
		return nil
	}

	err := backoff.Retry(operation, exp)
	if err != nil {
		log.Printf("[READY] Transcode was not ready after %d seconds - proceeding to mark as Ready now\n", int(exp.MaxElapsedTime.Seconds()))
		statsd.Inc(constTranscodeReadyFail, 1, 1.0)
	}

	var t usher.HlsTranscode
	t.Status = "active"
	t.ID = transcodeID
	t.Channel = *channel
	t.ChannelID = *channelID

	err = usherCfg.UpdateHlsEntry(transcodeID, t)
	if err != nil {
		log.Println("[READ] Failed to update usher hls entry", err)
	}

	err = MarkReady(isLvsChannel())
	if err != nil {
		log.Println("[READY] Failed to mark stream as ready")
	}

	log.Printf("[READY] Sending out StreamReady SNS notification for SessionID: %s\n", sessionID)
	//Send SNS notification
	err = notifyObj.NotifyStreamReady()
	if err != nil {
		log.Printf("[READY] Failed to push StreamReady SNS notification. Error: %v\n", err)
		statsd.Inc(fmt.Sprintf(constStatsSNSNotification, "stream_ready"), 1, 1.0)
	} else {
		log.Printf("[READY] Successfully sent StreamReady SNS notification for SessionID: %s\n", sessionID)
	}

	// Defer Stream Down SNS Notification on exit of transcoder
	onExit.Add(func() {
		log.Printf("[READY] Sending out StreamDown SNS notification for SessionID: %s\n", sessionID)
		err := notifyObj.NotifyStreamDown()
		if err != nil {
			log.Printf("[DOWN] Failed to push StreamDown SNS notification. Error: %v\n", err)
			statsd.Inc(fmt.Sprintf(constStatsSNSNotification, "stream_down"), 1, 1.0)
		} else {
			log.Printf("[DOWN] Successfully sent StreamDown SNS notification for SessionID: %s\n", sessionID)
		}
	})

}
