// Package adinsertion provides a Client which receives messages
// indicating that it's time to insert an ad into a stream.
package adinsertion

import (
	"encoding/base64"
	"fmt"
	"net/http"
	"net/url"
	"sync"

	"github.com/golang/protobuf/proto"
	"github.com/gorilla/websocket"

	"code.justin.tv/common/golibs/pubsubclient"
	"code.justin.tv/video/protocols/hlsext"
)

// Settings provides a description of the configuration needed to
// connect an AdInsertionClient.
type Settings struct {
	// URL of the pubsub edge to talk to. For example, darklaunch is at
	// "wss://pubsub-edge-darklaunch.twitch.tv:443/v1"
	URL string

	// HTTP Proxy to use when communicating to the pubsub edge. Empty
	// string means don't use any proxy.
	Proxy string

	// AuthToken to use when subscribing to the pubsub topic. On
	// Darklaunch, this can be "".
	AuthToken string

	// Topic returns the listening topic to hear ad breaks messages on
	Topic string
}

// A Client receives messages indicating a request for an ad break.
type Client interface {
	// Next blocks until an ad break is requested, or the pubsub
	// connection breaks (you'll get an error), or the client is closed
	// (you'll get nil, nil).
	Next() (*hlsext.AdBreakRequest, error)

	// Close ends the client's connection and unblocks Next().
	Close() error
}

// NewAdInsertionClient connects to a pubsub edge at url, subscribes
// to the ad request topic for channel, and returns a working
// Client. If it encounters an error dialing or subscribing, it
// returns the error and a nil client.
func NewAdInsertionClient(settings *Settings, channel string) (Client, error) {
	var proxy func(*http.Request) (*url.URL, error)
	if settings.Proxy != "" {
		url, err := url.Parse(settings.Proxy)
		if err != nil {
			return nil, fmt.Errorf("[AD INSERTION] unable to parse proxy URL: %s", err)
		}
		proxy = http.ProxyURL(url)
	}

	backend := pubsubclient.New(settings.URL, &websocket.Dialer{Proxy: proxy})
	return newPubsubClient(backend, fmt.Sprintf(settings.Topic, channel), settings.AuthToken)
}

// A pubsubClient implements the Client interface by communicating
// with pubsub.
type pubsubClient struct {
	client pubsubclient.Client

	sync.Mutex
	closed bool
}

func newPubsubClient(backend pubsubclient.Client, topic, authToken string) (*pubsubClient, error) {
	c := &pubsubClient{client: backend}
	err := c.client.Subscribe(topic, authToken)
	if err != nil {
		return nil, err
	}
	return c, nil
}

func (c *pubsubClient) Next() (*hlsext.AdBreakRequest, error) {
	msg, err := c.client.NextMessage()
	if err != nil {
		c.Lock()
		defer c.Unlock()
		if c.closed {
			return nil, nil
		}
		return nil, err
	}

	// Messages are base64-encoded because pubsub uses UTF-8 encoding of
	// messages, rewriting non-ASCII bytes into UTF-8 sequences, which
	// breaks the protobuf encoding.
	raw, err := base64.StdEncoding.DecodeString(msg.Message)
	if err != nil {
		return nil, fmt.Errorf("[AD INSERTION] base64 decode err: %s", err)
	}

	req := new(hlsext.AdBreakRequest)
	if err := proto.Unmarshal(raw, req); err != nil {
		return nil, fmt.Errorf("[AD INSERTION] proto unmarshal err: %s", err)
	}
	return req, nil
}

func (c *pubsubClient) Close() error {
	c.Lock()
	defer c.Unlock()

	if c.closed {
		return nil
	}
	c.closed = true

	return c.client.Close()
}
